ImplicitReplyActivity subclass: #ConcurrentActivity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Concurrent'!
ConcurrentActivity comment:
'Activity class ConcurrentActivity is an abstract class defining activity of concurrent objects, that is active objects having internal concurrency.
Usually it is necessary to control this concurrency, thus class ConcurrentActivity is defined as a subclass of activity class SelectiveAcceptActivity in order to control acceptance of messages.
Class ConcurrentActivity redefines acceptance of messages as the launching of a new subprocess to consume actual acceptance.

Note that there is no explicit control/synchronization of these sub activities as this is to be defined in specific subclasses expressing specific models of concurrency and synchronization.

One example is activity class SuspendActivity which defines quasi-concurrent objects which may launch several sub activities but with only one actually active at a time, other ones being suspended and waiting on synchronization conditions.
Another example is abstract activity class SynchroConcurrentActivity which defines synchronization generic events to be used in subclasses for example to update synchronization counters (see its subclass CountersActivity).'!


!ConcurrentActivity methodsFor: 'activity setting'!

startSubProcessFor: aMessage
	"Create and start a new subprocess to consume a message."
	"The message to be consumed is passed as an argument.
	This allowes lazy creation of subprocesses when needed (when message has been selected)."

	[self subProcessBodyFor: aMessage] newProcess resume!

subProcessBodyFor: aMessage
	"Define the subprocess body."
	"Default body is to accept and consume the message."

	self subProcessAcceptMessage: aMessage! !

!ConcurrentActivity methodsFor: 'message handling'!

acceptMessage: aMessage
	"Accept a message by launching a behavior subprocess to consume the message."

	self startSubProcessFor: aMessage!

subProcessAcceptMessage: aMessage
	"Acceptance of a message by a subprocess."
	"Default is standard acceptMessage: as for a primary process.
	Note we use super in order to refer to inherited definition of acceptMessage:
	because it is redefined within this class."

	super acceptMessage: aMessage! !

ConcurrentActivity subclass: #SynchroConcurrentActivity
	instanceVariableNames: 'synchroMutualExclusion '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Concurrent'!
SynchroConcurrentActivity comment:
'Activity class SynchroConcurrentActivity is an abstract class defining activity and synchronization of concurrent objects.
It is defined as a subclass of activity class ConcurrentActivity.

Class SynchroConcurrentActivity defines synchronization generic events.
They may be used in subclasses for example to update synchronization counters (see its subclass CountersActivity).

Instance variables:

	synchroMutualExclusion	<Semaphore>	ensures mutual exclusion of synchronization generic events.
Note this semaphore also ensures that success of checking acceptance of a message and synchronization event accept are atomic. Also associated class SynchroConcurrentAddress ensures that message enqueueing and synchronization event receive are atomic.'!


!SynchroConcurrentActivity methodsFor: 'initialize'!

privateInitialize
	"Initialize the semaphore for mutual exclusion of synchronization actions."

	super privateInitialize.
	synchroMutualExclusion := Semaphore forMutualExclusion! !

!SynchroConcurrentActivity methodsFor: 'accessing'!

synchroMutualExclusion
	^synchroMutualExclusion! !

!SynchroConcurrentActivity methodsFor: 'events'!

kernelEventComplete: aMessage
	"Dispatch onto the synchronization event complete which should be mutually exclusive with other ones."

	super kernelEventComplete: aMessage.
	synchroMutualExclusion critical:
		[self synchroEventComplete: aMessage]! !

!SynchroConcurrentActivity methodsFor: 'synchro events'!

synchroEventAccept: aMessage
	"This synchronization event method is called when the behavior accepts a message.
	It is used for synchronization purposes, for instance updating synchronization counters.
	At this abstract level, default behavior is empty."!

synchroEventComplete: aMessage
	"This synchronization event method is called once the behavior has completed computation of a message.
	It is used for synchronization purposes, for instance updating synchronization counters.
	At this abstract level, default behavior is empty."!

synchroEventReceive: aMessage
	"This synchronization event method is called when (BEFORE) the behavior receives a message.
	It is used for synchronization purposes, for instance updating synchronization counters.
	At this abstract level, default behavior is empty."! !

!SynchroConcurrentActivity methodsFor: 'default classes'!

addressClass
	"Needs to be paired with class SynchroConcurrentAddress
	which defines atomic synchronization event for message reception."

	^SynchroConcurrentAddress! !

!SynchroConcurrentActivity methodsFor: 'compatibility constraints'!

addressConstraint
	"Enforce assumption defined in method addressClass."

	^SynchroConcurrentAddress! !

ImplicitReplyAddress subclass: #SynchroConcurrentAddress
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Concurrent'!
SynchroConcurrentAddress comment:
'Address class SynchroConcurrentAddress is an adress class ensuring that message enqueueing and synchronization event receive are atomic.
It is used in association with activity class SynchroConcurrentActivity.'!


!SynchroConcurrentAddress methodsFor: 'message passing'!

asynchronousSend: aMessage inMessageQueue: queue
	"Ensure that asynchronous send (actual enqueueing of the message)
	AND the receive synchronization event are atomic."

	activity synchroMutualExclusion critical:
		[super asynchronousSend: aMessage inMessageQueue: queue.
		activity synchroEventReceive: aMessage]! !

!SynchroConcurrentAddress methodsFor: 'compatibility constraints'!

activityConstraint
	"SynchroConcurrentAddress method
		asynchronousSend:inMessageQueue:
			calls
	SynchroConcurrentActivity method
		synchroEventReceive:."

	^SynchroConcurrentActivity! !

