BoundedBuffer subclass: #IRBoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ex-BoundedBuffer-Ext'!
IRBoundedBuffer comment:
'Class IRBoundedBuffer (IR for implicit reply) defines a bounded buffer with a get request providing an implicit reply.'!


!IRBoundedBuffer methodsFor: 'script'!

get
	"Remove and return the first item.
	Do not check if buffer is empty."

	^contents removeFirst! !

!IRBoundedBuffer methodsFor: 'default classes'!

activityClass
	"A BoundedBuffer with implicit reply."

	^ImplicitReplyActivity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

IRBoundedBuffer class
	instanceVariableNames: ''!


!IRBoundedBuffer class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGet: numberItems! !

IRBoundedBuffer subclass: #GetRearBoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ex-BoundedBuffer-Ext'!


!GetRearBoundedBuffer methodsFor: 'script'!

getRear
	"Remove and return the last item.
	Do not check if buffer is empty."

	^contents removeLast! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GetRearBoundedBuffer class
	instanceVariableNames: ''!


!GetRearBoundedBuffer class methodsFor: 'example'!

exampleGetRearSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleGetRearSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleGetRearSize: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGetRear: numberItems! !

IRBoundedBuffer subclass: #Get2BoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ex-BoundedBuffer-Ext'!
Get2BoundedBuffer comment:
'This class implements an example of inheritance anomaly.'!


!Get2BoundedBuffer methodsFor: 'script'!

get2
	"Get two elements at once."

	^Array
		with: self get
		with: self get! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Get2BoundedBuffer class
	instanceVariableNames: ''!


!Get2BoundedBuffer class methodsFor: 'example'!

exampleGet2Size: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleGet2Size: 3 numberItems: 8 speedRatio: 3"
	"self exampleGet2Size: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGet2: numberItems! !

IRBoundedBuffer subclass: #GgetBoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ex-BoundedBuffer-Ext'!
GgetBoundedBuffer comment:
'This class implements an example of inheritance anomaly.'!


!GgetBoundedBuffer methodsFor: 'script'!

gget
	"Gget is like a get but should not be served after a get."
	"This constraint will be ensured by subclasses with various synchronization policies."

	^self get! !

!GgetBoundedBuffer methodsFor: 'events'!

eventAccept: aMessage
	"This trace may be useful to check that gget constraint (not after a put) is enforced
	by a specific synchronization policy."

	Transcript show: self printString , ' accept (' , aMessage compactPrintString , ')'; cr! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GgetBoundedBuffer class
	instanceVariableNames: ''!


!GgetBoundedBuffer class methodsFor: 'example'!

exampleGgetSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleGgetSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleGgetSize: 3 numberItems: 8 speedRatio: 1/3"

	"Note that there is no guarantee that all gget requests will be served.
	As a result the bounded buffer may stop before all producer and consumer requests had been served."

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGget: numberItems! !

ImplicitReplyObject subclass: #ConcurrentBoundedBuffer
	instanceVariableNames: 'array getIndex putIndex maxSize '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ex-BoundedBuffer-Ext'!
ConcurrentBoundedBuffer comment:
'Class ConcurrentBoundedBuffer implements a bounded buffer in a more basic way than class BoundedBuffer which is relying on an ordered collection.
The point is making put and get requests independent by managing different indexes. Thus this class may be used for concurrent accesses (allowing one put and one get at the same time).

Instance variables:

	array		<Array>		contains items.
	getIndex		<Integer>	index of next item to get.
	putIndex		<Integer>	index of next item to put.
	maxSize	<Integer>	max number of elements.'!


!ConcurrentBoundedBuffer methodsFor: 'initialize'!

initialize: anInteger
	maxSize := anInteger.
	"One more room is needed in order to discriminate between emptyness and fullness by comparing the two indexes."
	array := Array new: maxSize + 1.		
	getIndex := 1.
	putIndex := 1! !

!ConcurrentBoundedBuffer methodsFor: 'script'!

get
	| item |
	item := array at: getIndex.
	getIndex := (getIndex \\ array size) + 1.
	(Delay forSeconds: 2) wait.
	^item!

put: item
	array at: putIndex put: item.
	putIndex := (putIndex \\ array size) + 1.
	(Delay forSeconds: 2) wait! !

!ConcurrentBoundedBuffer methodsFor: 'state predicates'!

isEmpty
	^getIndex = putIndex!

isFull
	^(putIndex \\ array size) = (getIndex - 1)! !

!ConcurrentBoundedBuffer methodsFor: 'printing'!

printOn: aStream
	"Low level management of indexes. No need to look..."

	super printOn: aStream.
	aStream nextPut: $(.
	getIndex <= putIndex
		ifTrue:
			[getIndex to: putIndex - 1 do:
				[:i |
				(array at: i) printOn: aStream.
				aStream nextPut: $ ]]
		ifFalse:
			[getIndex to: array size do:
				[:i |
				(array at: i) printOn: aStream.
				aStream nextPut: $ ].
			1 to: putIndex -1 do:
				[:i |
				(array at: i) printOn: aStream.
				aStream nextPut: $ ]].
	aStream nextPut: $)! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ConcurrentBoundedBuffer class
	instanceVariableNames: ''!


!ConcurrentBoundedBuffer class methodsFor: 'instance creation'!

new: size
	^self new initialize: size! !

!ConcurrentBoundedBuffer class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGet: numberItems! !

ConcurrentBoundedBuffer subclass: #Get2ConcurrentBoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ex-BoundedBuffer-Ext'!
Get2ConcurrentBoundedBuffer comment:
'This class implements an example of inheritance anomaly.'!


!Get2ConcurrentBoundedBuffer methodsFor: 'script'!

get2
	"Get two elements at once."

	^Array
		with: self get
		with: self get! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Get2ConcurrentBoundedBuffer class
	instanceVariableNames: ''!


!Get2ConcurrentBoundedBuffer class methodsFor: 'example'!

exampleGet2Size: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleGet2Size: 3 numberItems: 8 speedRatio: 3"
	"self exampleGet2Size: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGet2: numberItems! !

ConcurrentBoundedBuffer subclass: #GetRearConcurrentBoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ex-BoundedBuffer-Ext'!
GetRearConcurrentBoundedBuffer comment:
'This class implements an example of inheritance anomaly.'!


!GetRearConcurrentBoundedBuffer methodsFor: 'script'!

getRear
	"Remove the last element of the buffer."

	putIndex := ((putIndex - 2) \\ (array size)) + 1.
	^array at: putIndex! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GetRearConcurrentBoundedBuffer class
	instanceVariableNames: ''!


!GetRearConcurrentBoundedBuffer class methodsFor: 'example'!

exampleGetRearSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleGetRearSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleGetRearSize: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGetRear: numberItems! !
