SingleMessageActiveObject subclass: #BoundedBufferClient
	instanceVariableNames: 'buffer delay '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ex-BoundedBuffer'!
BoundedBufferClient comment:
'Class BoundedBufferClient is defined as an abstract class to share common structure between producer and consumer classes.
Note that it is defined with a single message activity in order not to waiste further resources (infinite processes) during demonstrations of examples.'!


!BoundedBufferClient methodsFor: 'initialize'!

buffer: aBoundedBuffer delay: seconds
	buffer := aBoundedBuffer.
	delay := seconds! !

BoundedBufferClient subclass: #Producer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ex-BoundedBuffer'!
Producer comment:
'Class Producer implements the producer of a bounded buffer.
It is defined as a subclass of abstract class BoundedBufferClient.

Method runPut: runs the producer to issue a certain amount of put: (asynchronous) requests to the buffer.'!


!Producer methodsFor: 'script'!

runPut: max
	"Run max times a put: request to the bounded buffer."

	1 to: max do: [:i |
		Transcript show: 'Produces: ' , i printString; cr.
		buffer put: i.
		(Delay forSeconds: delay) wait]! !

BoundedBufferClient subclass: #Consumer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ex-BoundedBuffer'!
Consumer comment:
'Class Consumer implements the consumer of a bounded buffer.
It is defined as a subclass of abstract class BoundedBufferClient.

There are several methods to run the consumer, depending on the type of the communication protocols (asynchronous, synchronous, implicit reply) with the bounded buffer active object.

Additional run methods generate get2 (to get two elements at once) or gget (disabled after a put) requests. They are introduced in synchronization extension classes, see Actalk-Synchro-* categories. They have been proposed as seminal examples of inheritance anomalies by Satoshi Matsuoka and Akinori Yonezawa in the book Research Directions in COOP, MIT-Press, 1993.'!


!Consumer methodsFor: 'script'!

runGet2: max
	"Run max/2 times a get2 request to the bounded buffer (to remove max items).
	Assume the bounded buffer is an implicit reply active object."

	max / 2 timesRepeat:
		[Transcript show: 'Consumer requests two items'; cr.
		[Print reply: buffer get2 value] fork.
		(Delay forSeconds: delay) wait]!

runGet2: max replyTo: r
	"Run max/2 times a get2AndReplyTo: (asynchronous) request to the bounded buffer (to remove max items)."

	max / 2 timesRepeat:
		[Transcript show: 'Consumer requests two items'; cr.
		buffer get2AndReplyTo: r.
		(Delay forSeconds: delay) wait]!

runGet: max
	"Run max times a get (synchronous) request to the bounded buffer.
	Assume the bounded buffer is an implicit reply active object."

	max timesRepeat:
		[Transcript show: 'Consumer requests'; cr.
		[Print reply: buffer get value] fork.	"Do not wait onto the value."
		(Delay forSeconds: delay) wait]!

runGet: max replyTo: r
	"Run max times a getAndReplyTo: (asynchronous) request to the bounded buffer."

	max timesRepeat:
		[Transcript show: 'Consumer requests'; cr.
		buffer getAndReplyTo: r.
		(Delay forSeconds: delay) wait]!

runGetRear: max
	"Run max times alternatively get and getRear requests to the bounded buffer.
	Assume the bounded buffer is an implicit reply active object."

	| isGetRear |
	isGetRear := true.
	max timesRepeat:
		[Transcript show: 'Consumer requests '.
		isGetRear
			ifTrue:
				[Transcript show: 'getRear'; cr.
				[Print reply: buffer getRear value] fork]
			ifFalse:
				[Transcript show: 'get'; cr.
				[Print reply: buffer get value] fork].
		isGetRear := isGetRear not.
		(Delay forSeconds: delay) wait	]!

runGget: max
	"Run max times get and gget requests to the bounded buffer.
	Assume the bounded buffer is an implicit reply active object."
	"Temporary variable ratio defines the ratio between get and gget (1/ratio requests are gget)."

	| ratio nGet |
	ratio := 3.	"one third of requests are gget."
	nGet := 1.
	max timesRepeat:
		[Transcript show: 'Consumer requests '.
		nGet < ratio
			ifTrue:	
				[Transcript show: 'get'; cr.
				[Print reply: buffer get value] fork.
				nGet := nGet + 1]
			ifFalse:
				[Transcript show: 'gget'; cr.
				[Print reply: buffer gget value] fork.
				nGet := 1].
		(Delay forSeconds: delay) wait	]!

runGget: max replyTo: r
	"Run max times alternatively get and gget requests (asynchronously) to the bounded buffer."
	"Temporary variable ratio defines the ratio between get and gget (1/ratio requests are gget)."

	| ratio nGet |
	ratio := 3.	"one third of requests are gget."
	nGet := 1.
	max timesRepeat:
		[Transcript show: 'Consumer requests '.
		nGet < ratio
			ifTrue:	
				[Transcript show: 'get'; cr.
				buffer getAndReplyTo: Print.
				nGet := nGet + 1]
			ifFalse:
				[Transcript show: 'gget'; cr.
				buffer ggetAndReplyTo: Print.
				nGet := 1].
		(Delay forSeconds: delay) wait	]!

runIsEmptyGet2: max
	"Run max/2 times a get2 request to the bounded buffer (to remove max items).
	Also send isEmpty requests.
	Assume the bounded buffer is an implicit reply active object."

	max / 2 timesRepeat:
		[Transcript show: 'Consumer requests two items'; cr.
		[Print reply: buffer isEmpty value] fork.
		[Print reply: buffer get2 value] fork.
		(Delay forSeconds: delay) wait]! !

OrderedCollection variableSubclass: #BoundedBufferObject
	instanceVariableNames: 'maxSize '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ex-BoundedBuffer'!
BoundedBufferObject comment:
'Class BoundedBufferObject is a simple implementation of a bounded buffer internal and passive object as an ordered collection.

Instance Variables:

	maxSize	<Integer>	the maximum size of the bounded buffer
							(that is the maximum number of items it may contain).

Method addLast: is equivalent to a put:, and method removeFirst is equivalent to a get.
Method addLast: is redefined to check if the buffer is full. This is an internal check to trap whenever further synchronization constructs or programming is deficient.
State predicates (methods isFull and isEmpty) are also defined for convenience.'!


!BoundedBufferObject methodsFor: 'accessing'!

maxSize: anInteger
	maxSize := anInteger! !

!BoundedBufferObject methodsFor: 'adding'!

addLast: item
	"Catch and trigger an error if the buffer is full."
	"This provides (some) checking of synchronization constructs implementation, and specification."

	self size >= maxSize
		ifTrue: [self error: 'buffer is full']
		ifFalse: [super addLast: item]! !

!BoundedBufferObject methodsFor: 'state predicates'!

isEmpty
	^self size = 0!

isFull
	^self size = maxSize! !

ActiveObject subclass: #BoundedBuffer
	instanceVariableNames: 'contents maxSize '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ex-BoundedBuffer'!
BoundedBuffer comment:
'Class BoundedBuffer implements the seminal example of synchronization.

The point is following: put: requests should not be accepted while the buffer is full. Conversely get requests should not be accepted while the buffer is empty. The acceptance of the methods is depending on the state of the buffer, that is put: method should be disabled (in other words, put: messages should not be accepted) when it is full, and get method should be disabled when it is empty.

This is a seminal example of synchronization on state conditions. Here we don''t deal with intra-object concurrency where we would have to also control possible concurrent activations. See a first discussion of this point in class EmptyBoundedBuffer comment, in category Actalk-Actor-Examples. See examples of bounded buffer classes dealing with intra-object concurrency and activation synchronization for instance within category Actalk-Synchro-Counters-Ex.

In this example actual implementation of the bounded buffer relies on an internal bounded buffer object (class BoundedBufferObject) in order to focus on synchronization issues and not on data representation. In extensions with intra-object concurrency (e.g., category Actalk-Ex-BoundedBuffer-Ext) we will use an alternative implementation using an array and two indexes to clearly separate put and get accesses (accessing to distinct zones, a put and a get may take place concurrently).

In this first implementation (this example will be revisited in all kernel extensions dealing with synchronization), we face the absence of synchronization constructs because there is no way of controlling message acceptance. We can only acknowledge the failure in case of accepting disabled methods.

Instance Variables:

	contents		<BoundedBufferObject>	the bounded buffer object.
	maxSize	<Integer>				its maximum size, that is the maximum number of items.

The bounded buffer is shared by a producer (class Producer) and a consumer (class Consumer). It is possible to adjust relative speed of producer and consumer in order to check fullness or emptyness of the bounded buffer.
For instance using a ratio greater than 1 (e.g., 3 as in first example comment) will make the consumer work slower and as a consequence the bounded buffer will become full. A ratio lower than 1 (e.g., 1/3 as in second example comment) will start issueing get request onto an empty bounded buffer.

In all variations of class BoundedBuffer it may be specially useful to trace events to show the ordering of requests reception, acceptance and completion. You may set traces on a general class (for all example classes, evaluate ActiveObject setTrace: true) or more specific ones (ex: BoundedBuffer setTrace: true).

Note that the general issue is to be able to control the scheduling (starting of activation) of message invocations. Motivations may be to preserve consistency of the state (as for this BoundedBuffer example), avoid deadlock (see example of dining philosophers in category Actalk-Ext-Pool-Examples), avoid starvation, optimize services (for instance a printer would print out small jobs in priority), etc... By scheduling, we don''t mean that activations have to be serialized as in the case of objects with no intra-object concurrency.
The general problem is to find strategies to express such conditions to be general and expressive enough, modular enough to be easily inherited, and easy to implement efficiently. This general problem is still an open problem as we need to find best compromises/tradeoffs.'!


!BoundedBuffer methodsFor: 'initialize'!

initialize: anInteger
	maxSize := anInteger.
	contents := BoundedBufferObject new maxSize: maxSize! !

!BoundedBuffer methodsFor: 'script'!

getAndReplyTo: r
	"Remove and return the first item.
	Do not check if buffer is empty."

	r reply: contents removeFirst!

put: item
	"Add the item at the end.
	Do not check if buffer is full."

	contents addLast: item! !

!BoundedBuffer methodsFor: 'state predicates'!

isEmpty
	^contents isEmpty!

isFull
	^contents isFull! !

!BoundedBuffer methodsFor: 'printing'!

printOn: aStream
	"Print as default followed by its contents."

	super printOn: aStream.
	contents printOn: aStream! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

BoundedBuffer class
	instanceVariableNames: ''!


!BoundedBuffer class methodsFor: 'instance creation'!

new: maxSize
	^self new initialize: maxSize! !

!BoundedBuffer class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGet: numberItems replyTo: Print! !

BoundedBuffer subclass: #BoundedBuffer2
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ex-BoundedBuffer'!
BoundedBuffer2 comment:
'Class BoundedBuffer2 is a second version of the bounded buffer example.
It checks and traces when it cannot serve put or get requests.
It is defined as a subclass of class BoundedBuffer.'!


!BoundedBuffer2 methodsFor: 'script'!

getAndReplyTo: r
	"If empty, then reply an error acknowledgement."

	self isEmpty
		ifTrue:							"If empty,"
			[r reply: #getFailed]			"then, return a failure acknowledgement."
		ifFalse:							"Otherwise,"
			[super getAndReplyTo: r]		"serve the get."!

put: item
	"If full, then trace an error."

	self isFull
		ifTrue:							"If full,"
			[Print reply: #putFailed]		"then, print a failure acknowledgement."
		ifFalse:							"Otherwise,"
			[super put: item]				"serve the put:."! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

BoundedBuffer2 class
	instanceVariableNames: ''!


!BoundedBuffer2 class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	super exampleSize: maxSize numberItems: numberItems speedRatio: ratio! !

BoundedBuffer2 subclass: #BoundedBuffer3
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ex-BoundedBuffer'!
BoundedBuffer3 comment:
'Class BoundedBuffer3 is a first step towards management of synchronization.
It is defined as a subclass of class BoundedBuffer.

Rather than discarding methods which are disabled (e.g., put: method while the buffer is full), the buffer resends such requests to itself. This assumes that the unbounded (possibly infinite) cycle of reaccepting/resending the message will be eventually stopped (triggered by another request, e.g., a get request so the buffer could stop being full). This uses the method resend: (of class Basic2Address) which transparently ensures current behavior process yields way to others (because reaccept/resend infinite loop would keep the processor and forbid other active objects, like consumer to send other requests).

Notice that this strategy of resending messages breaks down the assumption of transmission message ordering. Some message which initially arrived before another one may finally be accepted after it because of the constant reordering within the mailbox. This may be a problem for some algorithms (e.g., class PrimeFilter) on which we would then need to add some timestamping strategy in order to recover the initial ordering at sending time. This may also be specified by synchronization constraints. See category Actalk-Synchro-Invocation as an example.
Such resending of message is also used for class NullBoundedBuffer (category Actalk-Ext-Actor-Examples) as well as for several classes to express synchronization. (To list them, find users of methods resend: and resendWithoutYield: defined in class Basic2Address).'!


!BoundedBuffer3 methodsFor: 'script'!

getAndReplyTo: r
	"If empty, then resend the message."

	self isEmpty
		ifTrue:							"If empty,"
			[aself resend:				"then, resend the original message..."
				(Message selector: #getAndReplyTo: argument: r)]	"...reconstructed."
		ifFalse:							"Otherwise,"
			[super getAndReplyTo: r]		"serve the get."!

put: item
	"If full, then resend the message."

	self isFull
		ifTrue:							"If full,"
			[aself resend:				"then, resend the original message..."
				(Message selector: #put: argument: item)]	"...reconstructed."
		ifFalse:							"Otherwise,"
			[super put: item]				"serve the put:."! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

BoundedBuffer3 class
	instanceVariableNames: ''!


!BoundedBuffer3 class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	super exampleSize: maxSize numberItems: numberItems speedRatio: ratio! !
