ActiveObject subclass: #MemoFibonacci
	instanceVariableNames: 'memoDictionary '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Examples'!
MemoFibonacci comment:
'Class MemoFibonacci computes fibonacci as a memofunction.
A memofunction is a function which memorizes its values once computed. This is specially useful for functions with may recompute many times same expressions. (This is the case for fibonacci because of the double recursive call).

Instance Variables:

	memoDictionary		<IdentityDictionary>	memorizes the associations <n,f(n)> once computed.

The memofunction first checks if the value is in the memory, otherwise it requests the actual computation with a new continuation. This continuation will dispatch the result of the actual computation to the initial reply destination and to the memofunction in order to memorize it.
MemoFibonacci cannot avoid recomputations if result of actual computation arrives too lately (which is mostly the case!!).
A more efficient and less naive version is class SmartMemoFunction which ensures only one actual computation.

Note that most of the class is generic. Therefore the improved version will be an abstract class: SmartMemoFunction.'!


!MemoFibonacci methodsFor: 'initialize'!

initialize
	memoDictionary := IdentityDictionary new! !

!MemoFibonacci methodsFor: 'script'!

at: n store: fn
	"Store the value of f(n) into the dictionary."

	memoDictionary at: n put: fn!

computeN: n replyTo: r
	"The actual computation. Exactly as for standard Fibonacci class."

	| jc |		"jc = join continuation."
	n < 2
		ifTrue: [r reply: 1]
		ifFalse:
			[jc := (BinaryAdder new r: r) active.
			aself n: n-1 replyTo: jc;
				n: n-2 replyTo: jc]!

n: n replyTo: r
	"Compute fibonacci by first looking into the memory.
	If not found request the actual computation
	and insert a continuation to memorize the result afterwards."

	| fn |
	fn := memoDictionary at: n ifAbsent: [#absent].
	fn = #absent									"If f(n) is found,"
		ifFalse: [r reply: fn]						"then, reply it."
		ifTrue: [aself computeN: n replyTo:			"Otherwise, we request the actual computation."
					(self singleReplyContinuationBlock: [:v :me |	"with a new continuation."
						r reply: v.
						aself at: n store: v])]		"which will also send the result for memorization."! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

MemoFibonacci class
	instanceVariableNames: ''!


!MemoFibonacci class methodsFor: 'instance creation'!

new
	^super new initialize! !

!MemoFibonacci class methodsFor: 'example'!

example
	"self example"

	"Inspect the expression to inspect the resulting memo dictionary."

	"Note that computation is actually much slower than simple Fibonacci
	because it does not avoid most of recomputations (results arrive too late!!).
	See SmartMemoFunction as a less naive version."

	^self new active
		n: 10 replyTo: Print! !

MemoFibonacci subclass: #SmartMemoFunction
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Examples'!
SmartMemoFunction comment:
'Class SmartMemoFunction is an improved version of memo functions.
It is defined as a subclass of class MemoFibonacci because it reuses the memoDictionary structure.
(This is because almost all definition of class MemoFibonacci was not specific to fibonacci computation).
Consequently we define SmartMemoFunction as an abstract class.
For simplification we just assume that the function has only one argument but it may be easily expanded to accept a variable number of arguments.
Concrete subclass SmartMemoFibonacci will just have to supply the semantics of the actual computation of the (fibonacci) function, through method computeN:replyTo:.

SmartMemoFunction ensures that actual computation of <n> occurs only once.
Requests for a computation of <n> are memorized (that is reply destinations are stored within the dictionary in place of the yet uncomputed value of f(n)). Once the value is computed it is stored in the dictionary and all pending reply destinations requesting this value are served.'!


!SmartMemoFunction methodsFor: 'script'!

at: n store: fn
	"Store the value of f(n) into the dictionary.
	If reply destinations requesting the value have been stored, reply the value to all of them."

	| vn |
	vn := memoDictionary at: n ifAbsent: [#absent].
	vn = #absent		"If there is an entry in the dictionary,"
					"then this is a collection of continuations waiting for (having requested) the value."
		ifFalse: [vn do: [:r | r reply: fn]].		"The value is sent to all of them."
	memoDictionary at: n put: fn		"(Then) the (only) assignment of f(n) takes place."

"Note that if there is a value in the dictionary, this CANNOT be the value f(n).
This is because the algorithm ensures that there is only one computation of f(n),
and consequently for each value of n ONLY ONE assigment of f(n) will take place.
Consequently the value is the collection of pending reply destinations having requested the value
 by message n:replyTo: and have been added to the collection."!

computeN: n replyTo: r
	"The actual computation must be specified by the concrete class."

	^self subclassResponsibility!

n: n replyTo: r
	"Compute the function by first looking into the memory.
	If not found the reply destination is added to the collection of requesting continuations
	which is stored into the memory."

	| vn |
	vn := memoDictionary at: n ifAbsent: [#absent].
	vn = #absent	
		ifTrue:
"If there is no entry in the dictionary, then, this means that this is the first consultation request.
Then create a collection for storing requesting reply destinations with current reply destination as first element."
			[memoDictionary at: n put: (OrderedCollection new add: r; yourself).
			aself computeN: n					"Then actual computation is requested,"
				replyTo: (self singleReplyContinuationBlock:	"with a new continuation inserted"
							[:v :me | aself at: n store: v])]	"which will send the result for memorization."
		ifFalse:
"Otherwise, an entry has been found.
This entry may be the actual value or a set of pending continuations requesting it."
			[(vn isKindOf: Number)
				ifTrue:	"If it is a numeric value, the requested value has already been memorized,"
					[r reply: vn]	"and then just reply it."
				ifFalse:	"Otherwise this is a collection of continuations."
					[(memoDictionary at: n) addLast: r]]	"then just add on the requesting continuation."! !

SmartMemoFunction subclass: #SmartMemoFibonacci
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Examples'!
SmartMemoFibonacci comment:
'Class SmartMemoFibonacci is a smart memofunction computing fibonacci.
It is defined as a subclass of class SmartMemoFunction. (See class SmartMemoFunction comment).
It only specifies the actual computation of fibonacci.'!


!SmartMemoFibonacci methodsFor: 'script'!

computeN: n replyTo: r
	"The actual computation. Exactly as for standard Fibonacci class."

	| jc |		"jc = join continuation."
	n < 2
		ifTrue: [r reply: 1]
		ifFalse:
			[jc := (BinaryAdder new r: r) active.
			aself n: n-1 replyTo: jc;
				n: n-2 replyTo: jc]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SmartMemoFibonacci class
	instanceVariableNames: ''!


!SmartMemoFibonacci class methodsFor: 'example'!

example
	"self example"

	"Inspect the expression to inspect the resulting memo dictionary."

	^self new active
		n: 10 replyTo: Print! !

ActiveObject subclass: #Factorial
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Examples'!
Factorial comment:
'Class Factorial computes the function factorial by using continuations. The consequence is that the same active object may compute concurrently several requests as shown by the example.

	fac(n) = fac(n-1) * n
		is split into two parts:
			the recursive call: fac(n-1)
			and the continuation: fac(n-1) * n

Class FactorialContinuation defines the computation of the latter part. This continuation is implemented as the reply destination of the recursive computation of fac(n-1). An alternative way is by using a block continuation (second version method n2:replyTo:).'!


!Factorial methodsFor: 'script'!

n2: n replyTo: r
	"Version2: uses a block continuation."

	n = 0
		ifTrue:
			[r reply: 1]
		ifFalse:
			[aself n: n-1 replyTo:
				(self singleReplyContinuationBlock: [:v :me |
					r reply: n * v])]!

n: n replyTo: r
	"Version1: uses an explicit class of continuation."

	n = 0
		ifTrue:
			[r reply: 1]
		ifFalse:
			[aself n: n-1 replyTo:
				(FactorialContinuation new n: n r: r) active]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Factorial class
	instanceVariableNames: ''!


!Factorial class methodsFor: 'example'!

example
	"self example"

	"The point is that fac(5) is printed before fac(10) although computation is started afterwards.
	 This proves that the two computations occur concurrently."

	self new active
		n: 10 replyTo: Print;
		n: 5 replyTo: Print!

example2
	"self example2"

	"Version2: uses a block continuation."

	self new active
		n2: 10 replyTo: Print;
		n2: 5 replyTo: Print! !

ActiveObject subclass: #MultiplyInRange
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Examples'!
MultiplyInRange comment:
'Class MultiplyInRange computes the function factorial in a concurrent way. (Class Factorial can compute several factorials concurrently but each one is processed serially).
This is a good example of dividing a recursive computation (basic example of divide and conquer strategy).
The starting is that computing fac(n) is equivalent to multiply all elements within interval [1 n].
Computation is divided in two sub-computation of two sub-intervals [1 mid] and [mid+1 n] and so on recursively.
For each division there is a join continuation (as for fibonacci, except that it will multiply the two values) created which will rejoin and multiply the two results.
While dividing intervals it will eventually reach a singleton interval ([i i]) which needs no further recursive computation (value is i).'!


!MultiplyInRange methodsFor: 'script'!

from: n to: p replyTo: r
	"Multiplies all integers within interval [n  p]."

	| jc mid |		"jc = join continuation; mid = middle of the interval [n  p]." 
	n = p
		ifTrue:
			[r reply: n]			"[n  n] = n"
		ifFalse:
			[jc := (Join2Multiplier new r: r) active.
			mid := (n+p)//2.
			self class new active	"computes [n  mid]"
				from: n to: mid replyTo: jc.
			self class new active	"computes [mid+1  p]"
				from: mid+1 to: p replyTo: jc]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

MultiplyInRange class
	instanceVariableNames: ''!


!MultiplyInRange class methodsFor: 'example'!

example
	"self example"

	"Computes factorial(10)."

	self new active
		from: 1 to: 10 replyTo: Print! !

ActiveObject subclass: #PrimeFilter
	instanceVariableNames: 'n next '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Examples'!
PrimeFilter comment:
'Class PrimeFilter computes successive prime numbers.
The algorithm uses an ordered chain of filter active objects, each filter representing some prime integer.
Successive integers are sent through the chain.
An integer is checked by a prime number. If it can be divided then it is rejected, otherwise it will be sent to next filter in the chain. If it reaches the end of the chain, this means that a new prime number has been discovered. Then a new filter is created and added onto the chain.
Testing of various numbers and creation of filters occur concurrently.

Note: the algorithm is correct thanks to one strong assumption (enforced by implementation of Actalk kernel), that is message ordering preservation. This means that if an active object sends successively two messages to another active object, that one will receive the two messages in the same order. (Otherwise the chain of prime numbers could be unordered resulting in losing the correctness of the algorithm.)

Instance Variables:

	n		<Integer>				the prime integer represented.
	next		<nil/PrimeFilter address>	next prime filter in the chain (or nil if the end).'!


!PrimeFilter methodsFor: 'initialize'!

n: anInteger
	n := anInteger! !

!PrimeFilter methodsFor: 'script'!

filter: i
	"Check if my value (n) divides i."

	i \\ n = 0						"If n is divided by i, then do nothing."
		ifFalse:
			[next isNil			"Otherwise,"
				ifTrue: 				"if the filter is the last of the chain,"
					[Print reply: i.		"this means that a new prime integer has been found,"
					next := (self class new n: i) active]	"and a new filter is added to the chain."
				ifFalse: [next filter: i]]	"Otherwise, the test is passed along to next filter in the chain."! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

PrimeFilter class
	instanceVariableNames: ''!


!PrimeFilter class methodsFor: 'example'!

checkUntil: max
	"self checkUntil: 100"

	| two |
	two := (self new n: 2) active.		"Create the first prime integer/filter in the chain."
	2 to: max do:						"Successive integers are sent into the chain."
		[:i | two filter: i]! !

SingleMessageActiveObject subclass: #FactorialContinuation
	instanceVariableNames: 'n r '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Examples'!
FactorialContinuation comment:
'Class FactorialContinuation defines the continuation of computation of factorial by class Factorial.
A more concise programming using block continuation active objects may also be found in class Factorial example.
Class FactorialContinuation is a subclass of class SingleMessageActiveObject because it assumes receiving only one reply (and this avoids wasting resources).

Instance Variables:

	n	<Integer>	the value of n, needed to complete the n * fac(n-1) computation.
	r	<Address>	the reply destination of the computation.
					The initial reply destination is the destination of the result, e.g., Print.'!


!FactorialContinuation methodsFor: 'initialize'!

n: anInteger r: aReplyDestination
	n := anInteger.
	r := aReplyDestination! !

!FactorialContinuation methodsFor: 'script'!

reply: v
	r reply: n * v! !

ActiveObject subclass: #Counter
	instanceVariableNames: 'contents '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Examples'!
Counter comment:
'Class Counter is the seminal (and most simple) example of (active) object.
It shows the four main differences with standard Smalltalk programming:

	Class Counter is defined as a subclass of class ActiveObject and not of class Object.

	The active object is actually created by sending the message active to the newly created behavior.

	Consulting the contents implies passing the reply destination (e.g., the predefined printer Print) because in standard Actalk asynchronous message passing there is no implicit reply. This also means that standard Smalltalk-80 caret (^) to return the value is of no use (except for escaping the method). Replying must be explicitly stated as a message sending (with reply: selector) to the reply destination, either passed as an argument or stored (for example by a continuation, see category Actalk-Kernel-JoinCont).

Note that all examples (should) have (at least) one example method, defined as a class method within protocol example (thus following standard Smalltalk-80 convention).

Instance Variables:

	contents		<Integer>	the contents of the counter.

In order to have homogeneous naming for methods passing reply destinations, we use the following conventions (in addition to using reply: as the convention for reply selector):

	add AndReplyTo: to the end of a selector when there is no other argument (besides the reply destination),
		example is selector contentsAndReplyTo:

	add replyTo: to the end of a selector when there are other arguments,
		example is selector n:replyTo: of class Factorial.

In case of passing the reply selector as a parameter, this becomes AndReplyTo:withSelector: or replyTo:withSelector:, example is selector contentsAndReplyTo:withSelector:.'!


!Counter methodsFor: 'initialize'!

contents: anInteger
	contents := anInteger! !

!Counter methodsFor: 'script'!

consultAndReplyTo: r
	"As for asynchronous message passing, returning reply needs to be made explicit.
	The reply destination (r) is passed as an argument."

	r reply: contents!

consultAndReplyTo: r withSelector: replySelector
	"If the reply selector is variable (e.g., reply: or store: or print:...),
	we need to pass it as an argument (replySelector) and compute it
	(method reply:withSelector: of class ActorBehavior)."

	r reply: contents withSelector: replySelector!

incr
	contents := contents + 1!

reset
	contents := 0! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Counter class
	instanceVariableNames: ''!


!Counter class methodsFor: 'example'!

example
	"self example"

	(self new contents: 100) active
				incr;
				incr;
				consultAndReplyTo: Print;
				incr;
				consultAndReplyTo: Print withSelector: #reply:! !

ActiveObject subclass: #Fibonacci
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Examples'!
Fibonacci comment:
'Class Fibonacci is similar with class Factorial except it produces a double recursive call. As a result there is a need to join the two sub-results fac(n-1) and fac(n-2). The continuation is therefore named a join continuation because it waits for several (in this case two) replies and rejoins the divided computation.

There are three basic ways to implement such additive join computations.

The first way, found in the category Actalk-Kernel-JoinCont, has to make some check to discriminate between the first replied value which needs being stored and the next one which triggers the operation. See method n:replyTo:.
The second way, found in category Actalk-Ext-Actor-JoinCont, uses the concept of behavior replacement of the actor computation model (see class ActorObject). It is specified in a cleaner way but which could lead to less efficient implementations. See method n3:replyTo:.
The third way uses block continuations (as for class Factorial). However programming is more tricky in the case of a join continuation (because it assigns a variable captured by the lexical block closure). See method n4:replyTo:.

There is a more sophisticated version of Fibonacci defined as a memo function (see classes MemoFibonacci and SmartMemoFibonacci).'!


!Fibonacci methodsFor: 'script'!

n2: n replyTo: r
	"Second version:
		Create recursively two new active objects to handle the two recursive sub computations."

	| jc |		"jc = join continuation."
	n < 2
		ifTrue:
			[r reply: 1]
		ifFalse:
			[jc := (BinaryAdder new r: r) active.
			self class new active n2: n-1 replyTo: jc.
			self class new active n2: n-2 replyTo: jc]!

n3: n replyTo: r
	"Third version:
		With a behavior replacement based join continuation adder."

	| jc |		"jc = join continuation."
	n < 2
		ifTrue:
			[r reply: 1]
		ifFalse:
			[jc := (ReplaceBinaryAdder new r: r) active.
			aself n3: n-1 replyTo: jc;
				n3: n-2 replyTo: jc]!

n4: n replyTo: r
	"Fourth version:
		With a block continuation adder.
	This one is a little more tricky!!
	The lexical block will assign the captured variable v1,
	declared as a temporary variable of the method, in order to store the initial value."

	| jc v1 |		"jc = join continuation."
	n < 2
		ifTrue:
			[r reply: 1]
		ifFalse:
			[jc := (self continuationBlock: [:v :me |
					v1 isNil	
						ifTrue:
							[v1 := v]
						ifFalse:
							[r reply: v1 + v.
							me terminate]]).		"Optimize resource recovery."
			aself n4: n-1 replyTo: jc;
				n4: n-2 replyTo: jc]!

n: n replyTo: r
	"First version:
		Send the two recursive sub computations requests to current active object with a shared join continuation."

	| jc |		"jc = join continuation."
	n < 2
		ifTrue:
			[r reply: 1]
		ifFalse:
			["First create a join continuation,"
			jc := (BinaryAdder new r: r) active.
			"then send to aself the two recursive messages, with the join continuation as the shared reply destination."
			aself n: n-1 replyTo: jc;
				n: n-2 replyTo: jc]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Fibonacci class
	instanceVariableNames: ''!


!Fibonacci class methodsFor: 'example'!

example
	"self example"

	self new active
		n: 10 replyTo: Print!

example2
	"self example2"

	"Second version:
		Create recursively two new active objects to handle the two recursive sub computations."

	self new active
		n2: 10 replyTo: Print!

example3
	"self example3"

	"Third version:
		With a behavior replacement based join continuation adder."

	self new active
		n3: 10 replyTo: Print!

example4
	"self example4"

	"Fourth version:
		With a block continuation adder."

	self new active
		n4: 10 replyTo: Print! !
