Abcl1Object subclass: #AbclVirtualBoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl-Examples'!
AbclVirtualBoundedBuffer comment:
'Class AbclVirtualBoundedBuffer implements the bounded buffer example by using the mailbox of the active object as the buffer itself.
The basic idea is to wait for and serve pairs of put/get requests. Single put or get requests will keep pending (being buffered) in the mailbox itself.

See also class ObjectBodyVirtualBoundedBuffer for a similar flavor but for a different activity model.'!


!AbclVirtualBoundedBuffer methodsFor: 'script'!

start
	"Keeps waiting for pairs get/put (get, then put)."
	"Needs to be sent to the active object as a starter."
	"Note that there is no script for methods put/get.
	They are only defined and accepted from within this body-like construct
	(see also class ObjectBodyVirtualBoundedBuffer)."

	[true] whileTrue:
		[self waitFor: #(getAndReplyTo:)
			andDo: [:r | self waitFor: #(put:)
						andDo: [:item | r reply: item]]]!

start2
	"Alternative to start. Keeps waiting for pairs put/get (put, then get).
	Equivalent behavior for clients."

	[true] whileTrue:
		[self waitFor: #(put:)
			andDo: [:item | self waitFor: #(getAndReplyTo:)
							andDo: [:r | r reply: item]]]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

AbclVirtualBoundedBuffer class
	instanceVariableNames: ''!


!AbclVirtualBoundedBuffer class methodsFor: 'example'!

example2Size: maxSize numberItems: numberItems speedRatio: ratio
	"self example2Size: 3 numberItems: 8 speedRatio: 3"
	"self example2Size: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := self new active start2.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGet: numberItems replyTo: Print!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := self new active start.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGet: numberItems replyTo: Print! !

Abcl1Object subclass: #SimulatedForwarder
	instanceVariableNames: 'forwardDest '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl-Examples'!
SimulatedForwarder comment:
'Please see class SimulatedNowSender comment.'!


!SimulatedForwarder methodsFor: 'initialize'!

forwardTo: anObject
	forwardDest := anObject! !

!SimulatedForwarder methodsFor: 'script'!

reply: value
	forwardDest reply: value! !

Abcl2Object subclass: #SimulatedNowReceiver
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl-Examples'!
SimulatedNowReceiver comment:
'Please see class SimulatedNowSender comment.'!


!SimulatedNowReceiver methodsFor: 'script'!

computeAndReplyTo: r
	(Delay forSeconds: 5) wait.
	r reply: #first! !

Abcl2Object subclass: #SimulatedNowSender
	instanceVariableNames: 'receiver '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl-Examples'!
SimulatedNowSender comment:
'Class SimulatedNowSender (and associated classes simulatedNowReceiver and SimulatedForwarder) show the reduction of ABCL/1 message types, that is reduction (here simulation) of a now type message send into:
	creation of a forwarder (to the sender object) which will be used as an identifier,
	past type message send with the forwarder as the reply destination,
	wait for the reply and check its sender, i.e., if it is the forwarder.
Original description of the reduction may be found in the OOCP book, pages 87-89.
In the example, the message selector is send. Another reply: message is sent before the actual reply to show the use of the where construct to identify the right reply.'!


!SimulatedNowSender methodsFor: 'initialize'!

initialize
	receiver := SimulatedNowReceiver new active! !

!SimulatedNowSender methodsFor: 'script'!

nowSend
	"Original now type message send to the receiver."

	| value |
	value := receiver computeAndReplyTo: #now.
	Transcript show: self printString , ' received waiting value: ' , value printString; cr!

pastSend
	"Past type message send to the receiver."

	receiver computeAndReplyTo: aself!

reply: value
	"Trying to fool the sender with another reply: message."

	Transcript show: self printString , ' received reply value: ' , value printString; cr!

simulatedNowSend
	"Simulated now type message send to the receiver."

	| newObject |
	"Create the identifier active object. It will forward the reply to myself."
	newObject := (SimulatedForwarder new forwardTo: aself) active.
	"Past type message send."
	receiver computeAndReplyTo: newObject.
	"Wait for the right reply."
	self waitFor: #(reply:) where: [:message | message sender = newObject] andDo: [:value |
		Transcript show: self printString , ' received waiting value: ' , value printString; cr]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SimulatedNowSender class
	instanceVariableNames: ''!


!SimulatedNowSender class methodsFor: 'instance creation'!

new
	^super new initialize! !

!SimulatedNowSender class methodsFor: 'example'!

exampleNow
	"self exampleNow"

	^self new active
		nowSend;
		reply: #second!

examplePast
	"self examplePast"

	^self new active
		pastSend;
		reply: #second!

exampleSimulatedNow
	"self exampleSimulatedNow"

	^self new active
		simulatedNowSend;
		reply: #second! !

Abcl1Object subclass: #TreeComparator
	instanceVariableNames: 'extractor1 extractor2 input1 input2 '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl-Examples'!
TreeComparator comment:
'TreeComparator implements pattern matching of the fringes of two trees (implemented as two Smalltalk arrays).
(This is the classical same-fringe example). This ABCL/1 version was originally described in OOPSLA''86 paper on ABCL/1. It was some derivation of algorithm introduced by Bernard Serpette in his thesis (1984) on concurrent Lisp (named Pive) with Unix-like pipes as communication channels.

There are three active objects in the algorithm.
Two extractors (TreeExtractor) extract the fringe of each tree, that is generate successive leaves.
The comparator (TreeComparator) compares the two fringes leaf by leaf.
The comparator send the initial requests to both extractors through two future type messages.
Each extractor will use its future object (MAFuture) reply destination to return each new leaf to the comparator.
In this example MAfutures are used as communication channels (as for Unix pipes).

Note that both extraction (by TreeExtractor) and comparison (by TreeComparator) methods are recursive and are both defined as private routines. TreeExtractor method extract: needs to be defined as a private routine because it involves recursion and synchronization. Therefore it cannot be defined in a straightforward way as a script method. (Remember that recursion and synchronous message sending leads to deadlock, thus we would have to find alternative non trivial way of writing down the extraction process).

In case of failure (matching fails) the comparator send messages in the express mode to both extractors in order to stop them (aborting their current tree extraction by using the nonResume construct).

Class TreeComparator is defined as a subclass of class Abcl1Object.
Class TreeExtractor is defined as a subclass of class Abcl3Object in order to manage express mode message send (for possible stop).'!


!TreeComparator methodsFor: 'initialize'!

initialize
	"Set the two tree extractors."

	extractor1 := (TreeExtractor new number: 1) active.
	extractor2 := (TreeExtractor new number: 2) active! !

!TreeComparator methodsFor: 'script'!

sameFringe: tree1 with: tree2
	"Compute a same fringe.
	Initiate the two communication channels through the two future type messages sends.
	Then start peeking into the two (queue) future objects used as two communication pipes."

	input1 := extractor1 fringe: tree1 replyTo: #future.
	input2 := extractor2 fringe: tree2 replyTo: #future.
	self compareNextLeaves! !

!TreeComparator methodsFor: 'private routines'!

compareNextLeaves
	"Compare leaves one by one after removing them from the two future (queue) objects."

	| leaf1 |
	leaf1 := input1 nextValue.
	(input2 nextValue) = leaf1
		ifTrue: [leaf1 = #EOT			"EOT = end of tree special symbol."
					ifTrue: [self success]
					ifFalse: [self compareNextLeaves]]
		ifFalse: [self failure]!

failure
	Print reply: false.
	"Stop and abort the two extractors with an express message."
	extractor1 ExpressStop.
	extractor2 ExpressStop!

success
	Print reply: true! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

TreeComparator class
	instanceVariableNames: ''!


!TreeComparator class methodsFor: 'instance creation'!

new
	^super new initialize! !

!TreeComparator class methodsFor: 'example'!

counterExample
	"self counterExample"

	self new active
		sameFringe:	#(1 (2 ((3) 40) 5) (6) (((7) 8) 9) ((10)))
		with:		#(1 2 3 4 5 6 7 8 9 10)!

example
	"self example"

	self new active
		sameFringe:	#(1 (2 ((3) 4) 5) (6) (((7) 8) 9) ((10)))
		with:		#(1 2 3 4 5 6 7 8 9 10)! !

Abcl3Object subclass: #ExpressCheck
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl-Examples'!
ExpressCheck comment:
'Class ExpressCheck is a simple check to show how express messages may interrupt, abort if unprotected standard (normal mode) message computation.'!


!ExpressCheck methodsFor: 'script'!

Express: delay
	"This is an express mode message."

	Transcript show: 'start express'; cr.
	(Delay forMilliseconds: delay) wait.
	Transcript show: 'express stop'; cr!

ExpressNonResume: delay
	"This is an express mode message."

	Transcript show: 'start express'; cr.
	(Delay forMilliseconds: delay) wait.
	Transcript show: 'nonResume '; cr.
	self nonResume.
	Transcript show: 'express stop'; cr!

normal: delay
	Transcript show: 'start normal'; cr.
	(Delay forMilliseconds: delay) wait.
	Transcript show: 'normal stop'; cr!

normalAtomic: delay
	self expressAtomic:
		[Transcript show: 'atomic '.
		self normal: delay]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ExpressCheck class
	instanceVariableNames: ''!


!ExpressCheck class methodsFor: 'example'!

example
	"self example"

	| o |
	o := self new active.
	o normal: 100; ExpressNonResume: 100; normalAtomic: 100; normal: 100.
	(Delay forMilliseconds: 1500) wait.
	o Express: 40! !

ActorCounter subclass: #AbclReplaceCounter
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl-Examples'!
AbclReplaceCounter comment:
'Class AbclReplaceCounter is an example of hybrid active object which has:

	activity and behavior semantics of the Actor model of computation,
	that is replacement behavior (active object behavior class ActorObject),
	and single message activity (activity class SingleMessageActivity),

	and:

	communication semantics of ABCL/1,
	that is the three types of message passing (address class AbclAddress).

This kybrid specie is working. Note that running the compatibility check returns true.
Meanwhile not all kinds of hybrids are necessary valid active objects.'!


!AbclReplaceCounter methodsFor: 'default classes'!

addressClass
	"Creating an hybrid kind of active object
		along actor computation model
		and with ABCL/1 3 types of message send."

	^Abcl1Address! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

AbclReplaceCounter class
	instanceVariableNames: ''!


!AbclReplaceCounter class methodsFor: 'example'!

example
	"self example"

	^(self new contents: 100) active
						incr;
						incr;
						consultAndReplyTo: #now! !

!AbclReplaceCounter class methodsFor: 'compatibility check'!

checkCompatibility
	"self checkCompatibility"

	^self checkConstraints! !

Abcl1Object subclass: #AbclBoundedBuffer
	instanceVariableNames: 'contents maxSize '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl-Examples'!
AbclBoundedBuffer comment:
'Class AbclBoundedBuffer implements the bounded buffer example with the waitFor:andDo: construct.
If a buffer accepted a disabled request (e.g., a put: message once it is full), it will explicitly wait for an alternative request (e.g., a get request) in order to first free the buffer.

See definition of put: and getAndReplyTo: methods.
Note that as opposed to behavior/state replacement as in class EmptyBoundedBuffer (category Actalk-Ext-Actor-Examples), synchronization code appears explicitly within the method body. This is because acceptance of method is controlled after already accepting a disabled request.
Note that mixing up synchronization code with method code is not highly modular. This proves to be a problem for reusing specifications in subclasses. As you may see in subclasses for get2 and gget methods.

Further note that there is an alternative version of the bounder buffer which uses its maibox as the buffer. See class AbclVirtualBoundedBuffer (and also ObjectBodyVirtualBoundedBuffer).'!


!AbclBoundedBuffer methodsFor: 'initialize'!

initialize: anInteger
	maxSize := anInteger.
	contents := BoundedBufferObject new maxSize: maxSize! !

!AbclBoundedBuffer methodsFor: 'script'!

getAndReplyTo: r
	"If empty, then wait for a put: and serve directly the get request."

	self isEmpty
		ifTrue: [self waitFor: #(put:)	 andDo: [:newItem |
					r reply: newItem]]
		ifFalse:
			[r reply: contents removeFirst]!

put: item
	"If full, then first wait for and serve a get."

	self isFull
		ifTrue: [self waitFor: #(getAndReplyTo:) andDo: [:r |
					r reply: contents removeFirst]].
	contents addLast: item! !

!AbclBoundedBuffer methodsFor: 'state predicates'!

isEmpty
	^contents isEmpty!

isFull
	^contents isFull! !

!AbclBoundedBuffer methodsFor: 'printing'!

printOn: aStream
	"Print as default followed by its contents."

	super printOn: aStream.
	contents printOn: aStream! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

AbclBoundedBuffer class
	instanceVariableNames: ''!


!AbclBoundedBuffer class methodsFor: 'instance creation'!

new: maxSize
	^self new initialize: maxSize! !

!AbclBoundedBuffer class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGet: numberItems replyTo: Print! !

AbclBoundedBuffer subclass: #AbclGet2BoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl-Examples'!
AbclGet2BoundedBuffer comment:
'This class implements an example of inheritance anomaly.'!


!AbclGet2BoundedBuffer methodsFor: 'script'!

get2AndReplyTo: r
	"If less than two items, wait for one put request.
	If buffer was not empty, then reply the stored item and the new one.
	Otherwise we need to wait for one more put."

	contents size < 2
		ifTrue:
			[self waitFor: #(put:) andDo: [:newItem |
				self isEmpty
					ifTrue:
						[self waitFor: #(put:) andDo: [:newItem2 |
							r reply: (Array
										with: newItem
										with: newItem2)]]
					ifFalse:
						[r reply: (Array
									with: contents removeFirst
									with: newItem)]]]
		ifFalse:
			[r reply: (Array
						with: contents removeFirst
						with: contents removeFirst)]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

AbclGet2BoundedBuffer class
	instanceVariableNames: ''!


!AbclGet2BoundedBuffer class methodsFor: 'example'!

exampleGet2Size: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleGet2Size: 3 numberItems: 8 speedRatio: 3"
	"self exampleGet2Size: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGet2: numberItems replyTo: Print! !

AbclBoundedBuffer subclass: #AbclGgetBoundedBuffer
	instanceVariableNames: 'afterPut '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl-Examples'!
AbclGgetBoundedBuffer comment:
'This class implements an example of inheritance anomaly.'!


!AbclGgetBoundedBuffer methodsFor: 'script'!

ggetAndReplyTo: r
	"Nothing to reuse and VERY complex.
	This is because if the buffer is empty, we need to wait for a put,
	but then we cannot accept the gget. So we need a put; put; get; gget sequence
	but it is not the only one valid!!"! !

Abcl3Object subclass: #TreeExtractor
	instanceVariableNames: 'output number '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl-Examples'!
TreeExtractor comment:
'See comment of class TreeComparator.'!


!TreeExtractor methodsFor: 'initialize'!

number: anInteger
	number := anInteger! !

!TreeExtractor methodsFor: 'script'!

ExpressStop
	"This is an express mode message."
	"Interrupt and abort current computation (method fringe:replyTo: and its subcomputation)."

	Transcript show: self printString , ' is stopped.'; cr.
	self nonResume!

fringe: tree replyTo: aReplyDestination
	"Initialize and start extraction of elements from the tree."

	"Store the reply destination, a future (queue) object, used as a (permanent) communication channel."
	output := aReplyDestination.
	Transcript show: self printString , ' extract: ' , tree printString; cr.
	self extract: tree.
	"EOT = end of tree special symbol. It is sent after completion."
	output reply: #EOT! !

!TreeExtractor methodsFor: 'private routines'!

extract: tree
	"Extraction of the leaves of the tree."
	"This is some standard recursive traversal of a tree.
	The representation of the tree is an array."

	"We use Processor yield in order to simulate simultaneity of computing between the two extractors.
	Note that generic event methods are not effective because this is a private routine."
	Processor yield.
	(tree isKindOf: Array)
		ifTrue:
			["A non leaf, that is a tree."
			tree isEmpty	
				ifFalse:
					["Recursive extraction."
					self extract: tree left;
						extract: tree right]]
		ifFalse:
			["A leaf."
			Transcript show: self printString , ' : ' , tree printString; cr.
			output reply: tree]! !

!TreeExtractor methodsFor: 'printing'!

printOn: aStream
	"Print as default followed by its number."

	super printOn: aStream.
	number printOn: aStream! !

Abcl1Object subclass: #AbclFibonacci
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl-Examples'!
AbclFibonacci comment:
'Class AbclFibonacci implements fibonacci computation with two variants.

The first one is using two waitFor:andDo: constructs in order to wait for both subresults, as proposed in ABCL/R2 (Masuhara et al. 1993).

The second one is adding two future type transmissions, as proposed in ABCL/f (Taura et al. 1994).
It is similar to IRFibonacci defined in category Actalk-Ext-ImplicitReply-Ex.'!


!AbclFibonacci methodsFor: 'script'!

n2: n replyTo: r
	"Second version:
		Create recursively two new active objects to handle the two recursive sub computations.
		Send two future type messages and sum up their values."

	n < 2
		ifTrue:
			[r reply: 1]
		ifFalse:
			[r reply:
				(self class new active n: n-1 replyTo: #future)
				+ (self class new active n: n-2 replyTo: #future)]!

n: n replyTo: r
	"First version:
		Create recursively two new active objects to handle the two recursive sub computations.
		Wait for both of their replies."

	n < 2
		ifTrue: [r reply: 1]
		ifFalse:
			[self class new active n: n-1 replyTo: aself.
			self class new active n: n-2 replyTo:aself.
			self waitFor: #(reply:) andDo: [:v1 |
				self waitFor: #(reply:) andDo: [:v2 |
					r reply: v1 + v2]]]! !

!AbclFibonacci methodsFor: 'default classes'!

addressClass
	"ABCL/f future type of message passing."

	^AbclfAddress! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

AbclFibonacci class
	instanceVariableNames: ''!


!AbclFibonacci class methodsFor: 'example'!

example
	"self example"

	"First version:
		Create recursively two new active objects to handle the two recursive sub computations.
		Wait for both of their replies."

	self new active
		n: 10 replyTo: Print!

example2
	"self example2"

	"Second version:
		Create recursively two new active objects to handle the two recursive sub computations.
		Send two future type messages and sum up their values."

	self new active
		n2: 10 replyTo: Print! !
