GenericSendAddress subclass: #Abcl1Address
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl'!
Abcl1Address comment:
'Class Abcl1Address implements the three ABCL/1 types of message passing:

	now	 type:		synchronous
	past	 type:		asynchronous, as in standard Actalk
	future type:		returning eagerly a future queue object (see class MAFuture)

On ABCL/1, see the book ABCL/1, edited by Akinori Yonezawa, MIT-Press, 1990.

Warning: recursive synchronous message passing immediately leads to a deadlock.
So don''t send a now message to aself!!

See examples in Actalk-Ext-Abcl-Examples.'!


!Abcl1Address methodsFor: 'message dispatching'!

receiveMessage: aMessage withSendType: type inMessageQueue: queue
	"Dispatch along the three types of message passing (now, past, future) of ABCL/1.
	Check last argument to be symbol #now or #future."

	type = #now
		ifTrue: [^self synchronousSend: aMessage inMessageQueue: queue].
	type = #future
		ifTrue: [^self futureSend: aMessage withFutureClass: MAFuture inMessageQueue: queue].
	^super receiveMessage: aMessage withSendType: type inMessageQueue: queue!

sendTypeOf: aMessage
	"Dispatch along the three types of message passing (now, past, future) of ABCL/1.
	Check last argument to be symbol #now or #future.
	Dispatch is done using sequence of tests with ^ escape/return (as for factorial method)."

	| lastArgument |
	aMessage arguments isEmpty
		ifTrue: [^#asynchronous].
	lastArgument := aMessage arguments last.
	(lastArgument isSymbol
			and: [#(now future) includes: lastArgument])
		ifTrue: [^lastArgument].
	^super sendTypeOf: aMessage! !

!Abcl1Address methodsFor: 'message passing types'!

insertReplyDestination: r intoMessage: aMessage
	"Replace the last argument by the reply destination.
	(Last argument is a symbol used for future type selection.)
	Return the modified message."

	aMessage arguments at: aMessage arguments size put: r.
	^aMessage! !

Abcl1Address subclass: #AbclfAddress
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl'!
AbclfAddress comment:
'Class AbclfAddress implements the single assignment future type of message passing of ABCL recent variant: ABCL/f.

It is defined as a subclas of class Abcl1Address.
Future type message passing dispatch is redefined to return a single assignment future object (class SAFuture and not MAFuture).

See example of use in class AbclFibonacci.'!


!AbclfAddress methodsFor: 'message dispatching'!

receiveMessage: aMessage withSendType: type inMessageQueue: queue
	"ABCL/f future type message passing creates a single assignment future object."

	type = #future
		ifTrue: [^self futureSend: aMessage withFutureClass: SAFuture inMessageQueue: queue].
	^super receiveMessage: aMessage withSendType: type inMessageQueue: queue! !

Abcl1Address subclass: #Abcl2Address
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl'!
Abcl2Address comment:
'Address class Abcl2Address assigns the sender of the message into the invocation. (It is analog to class WithSenderAddress.)
It is defined as a subclass of address class Abcl1Address.'!


!Abcl2Address methodsFor: 'message passing'!

receiveMessage: aMessage
	"Receive an extended invocation with the sender object address (reference)."

	^self receiveGenericMessage:
		((activity invocationClassFor: aMessage) new
				setSelector: aMessage selector arguments: aMessage arguments;
				sender: thisContext sender sender homeReceiver address)! !

!Abcl2Address methodsFor: 'compatibility constraints'!

activityConstraint
	"Abcl2Address method
		receiveMessage:
			assumes that the activity returns an invocation instance of class WithSenderInvocation
			(at least having one slot for sender)
			as defined by activity class
	WithSenderActivity."

	^WithSenderActivity! !

Abcl2Address subclass: #Abcl3Address
	instanceVariableNames: 'expressMailBox '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl'!
Abcl3Address comment:
'Class Abcl3Address implements the ABCL/1 express mode message passing.

Messages sent in the express mode take priority over normal mode messages (like an interruption).
Their selectors are tagged as starting with a E (E for Express).

Class Abcl3Address is defined as a subclass of class Abcl2Address.

See also associated classes of activity Abcl3Activity, for handling express mode messages, and of behavior Abcl3Object, for program control from and on express message activations.

Instance Variables:

	expressMailBox		<MailBox>	a mailqueue containing express messages.'!


!Abcl3Address methodsFor: 'initialize'!

privateInitialize
	"Initialize the express mailbox."

	super privateInitialize.
	expressMailBox := MailBox new! !

!Abcl3Address methodsFor: 'accessing'!

expressMailBox
	^expressMailBox! !

!Abcl3Address methodsFor: 'message passing'!

receiveMessage: aMessage withSendType: type withSendMode: mode
	"Dispatch along the express mailBox according to the express mode."

	mode = #express
		ifTrue: [^self receiveMessage: aMessage withSendType: type inMessageQueue: expressMailBox].
	^super receiveMessage: aMessage withSendType: type withSendMode: mode!

sendModeOf: aMessage
	"An express mode message is tagged through the first letter of the selector (E for express)."

	^(aMessage selector at: 1) = $E
		ifTrue: [#express]
		ifFalse: [super sendModeOf: aMessage]! !

WithSenderActivity subclass: #Abcl2Activity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl'!
Abcl2Activity comment:
'Activity class Abcl2Activity is a subclass of activity class WithSenderActivity.
It does not add nor redefine any functionality.
Its only purpose is to dispatch onto address class Abcl2Address, otherwise address classes specified by the behavior (Abcl2Address) and by the activity (WithSenderAddress) would be found incompatibles.'!


!Abcl2Activity methodsFor: 'default classes'!

addressClass
	"Abcl2Activity
		is defined in order to dispatch onto invocation class WithSenderInvocation
		which should be initialized by
	Abcl2Address method
		receiveMessage:."

	^Abcl2Address! !

!Abcl2Activity methodsFor: 'compatibility constraints'!

addressConstraint
	"Enforce assumption of method addressClass."

	^Abcl2Address! !

Abcl2Activity subclass: #Abcl3Activity
	instanceVariableNames: 'expressProcess expressExclusionSemaphore '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl'!
Abcl3Activity comment:
'Class Abcl3Activity is the class of activity to handle express mode messages.
A second activity process is created for handling express messages. When resumed (when an express message arrived), it suspends main activity process while computing an express message.

See behavior class Abcl3Object for program constructs controlling express mode message computation.

Instance Variables:

	expressProcess				<Process>		the process for computing express messages.
	expressExclusionSemaphore 	<Semaphore>	ensures mutual exclusion of express messages
												with the expressAtomic: construct.'!


!Abcl3Activity methodsFor: 'initialize'!

privateInitialize
	"Initialize the semaphore for ensuring atomicity against express message interruption."

	super privateInitialize.
	expressExclusionSemaphore := Semaphore forMutualExclusion! !

!Abcl3Activity methodsFor: 'accessing'!

expressExclusionSemaphore
	^expressExclusionSemaphore! !

!Abcl3Activity methodsFor: 'activity setting'!

start
	"Create and start a second activity process specific for express mode messages."

	super start.
	expressProcess :=
		[[true] whileTrue:
			[self acceptExpressMessage: address expressMailBox next]] newProcess.
	expressProcess resume! !

!Abcl3Activity methodsFor: 'process control'!

terminate
	"Terminate also express message activity process."

	"We have to launch a specific process to terminate
	both main activity process and express activity process.
	At first look it may seem very paradoxal indeed!!
	However the point is that we don't know what is current process:
	main activity process, or express activity process, or an external process.
	We have to terminate two processes in sequence.
	If the first process we terminate happens to be current process
	we would never terminate the second one!!"

	[super terminate.
	expressProcess terminateNoSignal] fork! !

!Abcl3Activity methodsFor: 'message handling'!

acceptExpressMessage: aMessage
	"Accepting an express message takes priority (suspend) over normal (standard) acceptance.
	Ensure mutual exclusion with the expressAtomic: construct."
	"Note that acceptance of express mode messages bypasses the generic events.
	Thus variable currentMessage set by event method kernelEventAccept: cannot be shadowed
	by an interrupting express mode message."

	expressExclusionSemaphore critical:
		[process suspend.
		self performMessage: aMessage.
		process resume]! !

!Abcl3Activity methodsFor: 'default classes'!

addressClass
	"Abcl3Activity method
		start
			calls
	Abcl3Address method
		expressMailBox."

	^Abcl3Address! !

!Abcl3Activity methodsFor: 'compatibility constraints'!

addressConstraint
	"Enforce assumption of method addressClass."

	^Abcl3Address! !

ActiveObject subclass: #Abcl1Object
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl'!
Abcl1Object comment:
'Class Abcl1Object implements the ABCL/1 wait-for construct.
Method waitFor:andDo: construct provides explicit acceptance of a message among a set of selectors. Once accepted the argument block is evaluated with the arguments of the message bound to the parameters of the block.
An example of the use of this construct is given in the bounded buffer example in category Actalk-Ext-Abcl-Example.

Note that in order to layer the large amount of functionalities and constructs offered by the ABCL/1 language, there are actually three progressive layers:

	Abcl1		wait for construct and 3 types of message send

	Abcl2		wait with where constraint construct, access to the sender of a message,

	Abcl3		express mode of message send.'!


!Abcl1Object methodsFor: 'wait construct'!

waitFor: arrayOfSelectors andDo: aBlock
	"Look into the mailbox for the first message matching one of the selectors,
	and then accept it.
	It is blocking and suspends until it finds some matching message.
	(This means it suspends onto incoming messages as long as necessary)."

	^aBlock valueWithArguments:
		(aself mailBox firstMessageWithCondition: [:message |
				arrayOfSelectors includes: message selector])
			arguments! !

!Abcl1Object methodsFor: 'default classes'!

addressClass
	"Usually used paired with an ABCL/1 address providing the 3 types of message passing."

	^Abcl1Address! !

Abcl1Object subclass: #Abcl2Object
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl'!
Abcl2Object comment:
'Class Abcl2Object defines where constraints on acceptance of messages and provides also information about the sender of a message.

Within a selective message acceptance, the where constraint acts as an additive constraint. Messages which do not match pattern and condition requirements are kept within the mailbox.
Method waitFor:where:andDo: is analog to method waitFo:andDo: plus the additional where constraint.

Management of the sender of a message is analog to class WithSenderObject (and its related activity and address classes).

Class Abcl2Object is a subclass of class Abcl1Object.'!


!Abcl2Object methodsFor: 'sender accessing'!

sender
	"Sender of current message."

	^self activity currentMessage sender! !

!Abcl2Object methodsFor: 'wait construct'!

waitFor: arrayOfSelectors where: conditionBlock andDo: aBlock
	"Look into the mailbox for the first message matching one of the selectors
	AND satisfying the condition (a block whose first argument is the candidate message).
	Then accept it.
	It is blocking and suspends until it finds some matching message.
	(This means it suspends onto incoming messages as long as necessary)."

	^aBlock valueWithArguments:
		(aself mailBox firstMessageWithCondition: [:message |
				(arrayOfSelectors includes: message selector)
					and: [conditionBlock value: message]])
			arguments! !

!Abcl2Object methodsFor: 'default classes'!

activityClass
	"Abcl2Activity activity class purpose is to dispatch onto address class Abcl2Address,
	otherwise address classes specified by the behavior (Abcl2Address)
	and by the activity (WithSenderAddress) would be found incompatibles."

	^Abcl2Activity!

addressClass
	"Abcl2Object method
		sender
			assumes that the sender has been set by
	Abcl2Address method
		receiveMessage:."

	^Abcl2Address! !

!Abcl2Object methodsFor: 'compatibility constraints'!

activityConstraint
	"Enforce assumption of method activityClass."

	^Abcl2Activity!

addressConstraint
	"Enforce assumption of method addressClass."

	^Abcl2Address! !

Abcl2Object subclass: #Abcl3Object
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Abcl'!
Abcl3Object comment:
'Class Abcl3Object defines program constructs to control express message computation.

Method nonResume may be used within an express message method. It will abort resumption of computing a normal mode message if one was being interrupted before its completion.
Method expressAtomic: may be used within any standard method. It ensures that computation won''t be interrupted by a possibly incoming express message.

Class Abcl3Object is a subclass of class Abcl2Object.'!


!Abcl3Object methodsFor: 'express method construct'!

expressAtomic: aBlock
	"Ensure that computation expressed within aBlock won't be interrupted by an express message.
	May be used within script methods as well as private routines."
	
	"WARNING: This construct should NOT be used within an express method,
	otherwise the express method computation will DEADLOCK!!!!
	Also express atomic blocks cannot be embedded otherwise it would also result in a deadlock.
	This is a current limitation of this simple simulation/implementation."

	self activity expressExclusionSemaphore critical: aBlock! !

!Abcl3Object methodsFor: 'express control construct'!

nonResume
	"Terminate current standard activity process and recreate a new one.
	In case a normal message was being processed, it will be aborted."

	"Should be used ONLY by express message methods!!!!
	(Otherwise if used within standard methods,
	it would abort main activity process which is current process!!!!)."

	"Note: the new activity process will be only created and not started,
	so that it will be in the suspend state as was current activity process to be resumed."

	self activity process terminateNoSignal.
	self activity createProcess! !

!Abcl3Object methodsFor: 'default classes'!

activityClass
	"Abcl3Object method
		expressAtomicBlock:
			calls
	Abcl3Activity method
		expressExclusionSemaphore ."

	^Abcl3Activity! !

!Abcl3Object methodsFor: 'compatibility constraints'!

activityConstraint
	"Enforce assumption of method activityClass."

	^Abcl3Activity! !
