ImplicitReplyObject subclass: #QuickSortTree
	instanceVariableNames: 'leftSubTree pivot rightSubTree '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Accept-Examples'!
QuickSortTree comment:
'Class QuickSortTree implements a distributed sorter.
This simple algorithm has been initially described in ABCL/1 by Etsuya Shibayama and Akinori Yonezawa in Distributed Computing in ABCL/1, in the book OOCP, edited by A. Yonezawa and M. Tokoro, MIT-Press, 1987, pages 123-126. This divide and conquer algorithm regards the data as active objects and requests as messages. This makes the synchronization easy as opposed as if we would let concurrent processes access a passive tree.
The initial algorithm has been someway transformed and simplified here.

Each QuickSortTree is recursively decomposed into a pivot value and two subtrees, left and right, according to elements lower or greater than the pivot.
Note that the method insert:andAckTo: specifies a reply destination to which an acknowledgement message (oneInserted) will be issued. The client (QuickSortClient) will accept (wait for) as many oneInserted messages as there are elements to be sorted. Therefore the client will be able to know when the sorting is completed.

Class QuickSort is a subclass of class ImplicitReplyObject, as it uses synchronous message passing to recursively retrieve elements.

Instance variables:

	leftSubTree		<QuickSortTree address>		left sub tree.
	pivot			<Object>					the division pivot value stored.
	rightSubTree	<QuickSortTree address>		right sub tree.

See also class QuickSortClient comment.
See also a similar example of a distributed symbol table in category Actalk-Ext-Pool-Examples.'!


!QuickSortTree methodsFor: 'script'!

insert: value andAckTo: r
	"Insert an element into this quicksort tree and acknowledge to the reply destination (client).
	Insertion is selected between both sub trees according to the pivot value."

	pivot isNil
		ifTrue:
			["If no pivot value yet (unitialized tree), then insert the value as the new pivot value."
			pivot := value.
			"Acknowledge that one element has been inserted."
			r oneInserted]
		ifFalse:
			["Otherwise, insertion is delegated to one of the sub trees.
			Note that the sub trees are lazily created."
			value < pivot
				ifTrue:
					[leftSubTree isNil
						ifTrue: [leftSubTree := self class new active].
					leftSubTree insert: value andAckTo: r]
				ifFalse:
					[rightSubTree isNil
						ifTrue: [rightSubTree := self class new active].
					rightSubTree insert: value andAckTo: r]]!

retrieveAndAddTo: aCollection
	"Retrieve (in order) the sorted elements by adding them to a collection.
	Note the synchronization (by waiting for the value of each recursive call) in order to ensure ordering."

	leftSubTree isNil
		ifFalse: [(leftSubTree retrieveAndAddTo: aCollection) value].
	aCollection add: pivot.
	rightSubTree isNil
		ifFalse: [(rightSubTree retrieveAndAddTo: aCollection) value]! !

ExplicitAcceptObject subclass: #QuickSortClient
	instanceVariableNames: 'topTree n '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Accept-Examples'!
QuickSortClient comment:
'ClassQuickSortClient is used to control a distributed quick sort (QuickSortTree).
Class QuickSortClient is defined as a subclass of class ExplicitAcceptObject, as it uses the answer: construct to determine when sorting is completed (by accepting n messages oneInserted acknowledged by the QuickSortTree).
In order to display the result in a comprehensive form, it retrieves the elements once sorted and stored in a distributed way in the QuickSortTree, and recollects them into a collection.

Instance variables:

	topTree	<QuickSortTree address>		the top quick sort tree which will sort and contain elements.
	n		<Integer>					number of elements to sort.

See also class QuickSortTree comment.'!


!QuickSortClient methodsFor: 'script'!

oneInserted
	"Do nothing.
	Used for waiting for n elements to be inserted by accepting n messages oneInserted."!

retrieve
	"Retrieve the elements sorted into a new collection and return it."

	| aCollection |
	aCollection := OrderedCollection new.
	(topTree retrieveAndAddTo: aCollection) value.
	Transcript show: 'Result: ' , aCollection printString; cr.
	^aCollection!

sort: aCollection
	"Sort the elements within the collection. Retrieve the result into a new collection."

	Transcript show: 'Sort: ' , aCollection printString; cr.
	topTree := QuickSortTree new active.
	n := aCollection size.
	aCollection do: [:value |
		"Insert all elements."
		topTree insert: value andAckTo: aself].
	n timesRepeat:
		"Wait for completion of insertion/sorting."
		[self answer: #(oneInserted)].
	Transcript show: 'Sort done.'; cr.
	^self retrieve! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

QuickSortClient class
	instanceVariableNames: ''!


!QuickSortClient class methodsFor: 'example'!

example
	"self example"

	^(self new active
		sort: #(6 2 10 7 4 8 1 5 9 3)) value!

example2
	"self example2"

	"This can sort integers but also all types of objects with some order relation (message >)."

	^(self new active
		sort: #(a c t a l k	i s	w i n n i n g	a g a i n !!)) value!

example: size
	"self example: 100"

	"Sort a random collection of a certain size."

	| aCollection random |
	aCollection := OrderedCollection new.
	random := Random new.
	size timesRepeat:
		[aCollection add: (random next * size) rounded].
	^self new active
		sort: aCollection! !

ExplicitAcceptObject subclass: #AcceptBoundedBuffer
	instanceVariableNames: 'contents maxSize '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Accept-Examples'!
AcceptBoundedBuffer comment:
'Class AcceptBoundedBuffer implements the bounded buffer example with the answer: construct for explicit message acceptance.
If a buffer accepted some disabled request (e.g., a put: message once it is full), it will explicitly wait for an alternative request (e.g., a get request) in order to first free the buffer.

See definition of put: and get methods.

Note that as for class AbclBoundedBuffer, synchronization code appears explicitly within the method body. As a result this model is not very good at reusing specifications.'!


!AcceptBoundedBuffer methodsFor: 'initialize'!

initialize: anInteger
	maxSize := anInteger.
	contents := BoundedBufferObject new maxSize: maxSize! !

!AcceptBoundedBuffer methodsFor: 'script'!

get
	"If empty, then first accept a put:."

	self isEmpty
		ifTrue: [self answer: #(put:)].
	^contents removeFirst!

put: item
	"If full, then first accept a get."

	self isFull
		ifTrue: [self answer: #(get)].
	contents addLast: item! !

!AcceptBoundedBuffer methodsFor: 'state predicates'!

isEmpty
	^contents isEmpty!

isFull
	^contents isFull! !

!AcceptBoundedBuffer methodsFor: 'printing'!

printOn: aStream
	"Print as default followed by its contents."

	super printOn: aStream.
	contents printOn: aStream! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

AcceptBoundedBuffer class
	instanceVariableNames: ''!


!AcceptBoundedBuffer class methodsFor: 'instance creation'!

new: size
	^self new initialize: size! !

!AcceptBoundedBuffer class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGet: numberItems! !

AcceptBoundedBuffer subclass: #AcceptGet2BoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Accept-Examples'!
AcceptGet2BoundedBuffer comment:
'This class implements an example of inheritance anomaly.'!


!AcceptGet2BoundedBuffer methodsFor: 'script'!

get2
	"If less than two items, wait for one or two (as needed) put request(s)."

	contents size < 2
		ifTrue:
			[self answer: #(put:).
			contents size = 1
				ifTrue: [self answer: #(put:)]].
	^Array
		with: self get
		with: self get! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

AcceptGet2BoundedBuffer class
	instanceVariableNames: ''!


!AcceptGet2BoundedBuffer class methodsFor: 'example'!

exampleGet2Size: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleGet2Size: 3 numberItems: 8 speedRatio: 3"
	"self exampleGet2Size: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGet2: numberItems! !

AcceptBoundedBuffer subclass: #AcceptGgetBoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Accept-Examples'!
AcceptGgetBoundedBuffer comment:
'This class implements an example of inheritance anomaly.'!


!AcceptGgetBoundedBuffer methodsFor: 'script'!

gget
	"Nothing to reuse and VERY complex.
	Same problem as for Abcl waitFor construct."! !

