ActorObject subclass: #ActorCounter
	instanceVariableNames: 'contents '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Actor-Examples'!
ActorCounter comment:
'Class ActorCounter is the actor computation model version of the example of a counter.
The difference with class Counter is that there is no assignment of instance variable contents but replacement of behavior.

Instance Variables:

	contents		<Integer>	the contents of the counter.'!


!ActorCounter methodsFor: 'accessing'!

contents: anInteger
	contents := anInteger! !

!ActorCounter methodsFor: 'script'!

consultAndReplyTo: r
	self ready.		"Note that a replacement is necessary although behavior remains constant."
	r reply: contents!

incr
	self replace: (self class new contents: contents + 1)!

reset
	self replace: (self class new contents: 0)! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ActorCounter class
	instanceVariableNames: ''!


!ActorCounter class methodsFor: 'example'!

example
	"self example"

	(self new contents: 100) active
				incr;
				incr;
				consultAndReplyTo: Print;
				incr;
				consultAndReplyTo: Print! !

ActorObject subclass: #BankAccount
	instanceVariableNames: 'balance '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Actor-Examples'!
BankAccount comment:
'Class BankAccount represents the seminal example of actors along the actor computation model.
It is very similar in spirit to class ActorCounter. Extensions to manage overdrafts may be more of interest (see class ProtectedBankAccount in category Actalk-Ext-Actor-Ext-Ex).

Note that when specifying a replacement behavior, either the behavior is explicitly named (in that case BankAccount) or implicitly refered to (self class). The latter case is more generic in that the methods may be reused in subclasses.

Instance Variables:

	balance		<Integer>	the balance of the bank account.'!


!BankAccount methodsFor: 'accessing'!

balance: anInteger
	balance := anInteger! !

!BankAccount methodsFor: 'script'!

balanceAndReplyTo: r
	self ready.
	r reply: balance!

deposit: amount replyTo: r
	self replace: (self class new balance: balance + amount).
	r reply: #depositReceipt!

withdraw: amount replyTo: r
	amount < balance
		ifTrue:
			["Safe case."
			self replace: (self class new balance: balance - amount).
			r reply: #withdrawalReceipt]
		ifFalse:
			["Overdraft. We leave the balance unchanged..."
			self ready.
			" ...before taking action... (Here we take NO action!!).
			See class ProtectedBankAccount for a real overdraft management."
			r reply: #overdraftReceipt]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

BankAccount class
	instanceVariableNames: ''!


!BankAccount class methodsFor: 'example'!

example
	"self example"

	(self new balance: 200) active
		deposit: 150 replyTo: Print;
		withdraw: 400 replyTo: Print		"Create an overdraft (50)."! !

ActorObject subclass: #NullBoundedBuffer
	instanceVariableNames: 'contents maxSize '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Actor-Examples'!
NullBoundedBuffer comment:
'Class NullBoundedBuffer implement a null (maximum size is 0) bounded buffer.
See class EmptyBoundedBuffer comment for further explanations.

Although not very useful in practice, it is useful as an abstract class for behavior classes EmptyBoundedBuffer and FullBoundedBuffer.
Therefore it implements redefinition of method doesNotUnderstand: to resend disabled/unknown messages (see class EmptyBoundedBuffer comment).
It also implements initialization and printing methods.
It is defined as a subclass of class ActorObject.

Instance Variables:

	contents		<BoundedBufferObject>	the bounded buffer object.
	maxSize	<Integer>				its maximum size, that is the maximum number of items.'!


!NullBoundedBuffer methodsFor: 'accessing'!

contents: aBoundedBufferObject maxSize: anInteger
	contents := aBoundedBufferObject.
	maxSize := anInteger! !

!NullBoundedBuffer methodsFor: 'message handling'!

doesNotUnderstand: aMessage
	"Implement delaying of disabled methods by trapping error message and resending the original message.
	Note: this means that any unknown method is a disabled method."

	"Behavior replacement (self ready) should not be forgotten as for any message acceptance.
	Otherwise the actor cannot accept next message.
	Note that Processor yield is unnecessary because this process terminates anyway."

	self ready.
	aself resendWithoutYield: aMessage! !

!NullBoundedBuffer methodsFor: 'state predicates'!

isEmpty
	^contents isEmpty!

isFull
	^contents isFull! !

!NullBoundedBuffer methodsFor: 'printing'!

printOn: aStream
	"Print as default followed by its contents."

	super printOn: aStream.
	contents printOn: aStream! !

NullBoundedBuffer subclass: #FullBoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Actor-Examples'!
FullBoundedBuffer comment:
'Class FullBoundedBuffer represents a full bounded buffer.
It is defined as a subclass of class NullBoundedBuffer.
It only defines method getAndReplyTo:.'!


!FullBoundedBuffer methodsFor: 'script'!

getAndReplyTo: r
	"This is a full buffer, thus it defines and may accept a get."

	r reply: contents removeFirst.
	self replace: (self isEmpty		"Compute replacement behavior."
					ifTrue: [EmptyBoundedBuffer new contents: contents maxSize: maxSize]
					ifFalse: [PartialBoundedBuffer new contents: contents maxSize: maxSize])! !

NullBoundedBuffer subclass: #EmptyBoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Actor-Examples'!
EmptyBoundedBuffer comment:
'Class EmptyBoundedBuffer (with classes PartialBoundedBuffer and FullBoundedBuffer) implements the bounded buffer in the actor model of computation.

As for class BoundedBuffer3 (in category Actalk-Ex-BoundedBuffer), a bounded buffer will resend requests for methods which are disabled. But as opposed to class BoundedBuffer3 such resending is not written within the body of methods. This is because behavior replacement may already explicitly specify which methods are enabled (and it implicitly disables any other one). Therefore there is not a single class for bounded behavior behavior but three of them. They correspond to the three possible states of a bounded buffer and associated enabled methods. Empty only enables put:. Full only enables get. Partial enables both of them. When accepting a message the replacement behavior is computed depending on the resulting state (empty, partial, full) of the bounded buffer.

Note that if the bounded buffer maximum size is 1, the partial state behavior will never be used.

In the (limit) case where the buffer is created with a maximum size of 0, there should be a specific behavior class disabling both put: and get. Although this case is of no use in practice, we may encounter it. Class NullBoundedBuffer defines such behavior. It defines everything necessary for a bounded buffer except accessing methods put: and get!! However this class is also useful as an abstract/sharing superclass for classes EmptyBoundedBuffer and FullBoundedBuffer.

Therefore class EmptyBoundedBuffer is defined as a subclass of class NullBoundedBuffer.
Class EmptyBoundedBuffer only defines method put:.

Because disabled methods are simply unknown methods, we simply trap the doesNotUnderstand: error message and resend the original message. This means that any unknown method is a disabled method and that any wrong (e.g., mispelled) message would be indefinitely delayed.

After the initial step of class BoundedBuffer3 to implement synchronization, this further step is more modular (enabling methods is now explicitly stated through the behavior class. Methods do not include any synchronization code). But it still relies on resending of a message. (This consumes resources including time, and this breaks down the assumption of transmission ordering preservation).
Easy possible improvement is to control (whether a method is enabled or not) BEFORE message acceptance, in order to avoid such resending and to recover the assumption of transmission ordering preservation. See activity class EnabledSelectorsActorActivity in category Actalk-Ext-Actor-Ext, and resulting class EmptyBoundedBuffer2 in category Actalk-Ext-Actor-Ext-Ex.

This concept of enabled methods associated to the state of the behavior may also be expressed apart from the behavior replacement model, that is for standard active objects. This results in the concept of abstract state (as the set of enabled methods). See class AbstractStatesActivity within category Actalk-Synchro-AbsStates.

Note that although the behavior replacement concept allowes intra-object concurrency, here we cannot specify in a natural way that one get and one put: requests could be processed concurrently while disallowing concurrent put: or concurrent get. Let''s take the example of a partial bounded buffer which accepts a put: request. It could concurrently accept a get request, but not another put until it has completed current put. If we specify a FullBoundedBuffer as immediate replacement behavior (although the buffer is not necessary becoming full) we temporarily disallow other put request, but within computation of further replacement behavior we have no way of checking completion of current put method, which is THE condition to accept another put method. In other words, in the bounded buffer example we have to deal with synchronization on state conditions as we did, but we also should deal with conditions for activation (one put a most and one get at most). The behavior replacement model can nicely specify state synchronization but is not good at activation conditions, although it may create intra-object concurrency. In the case of models with no intra-object concurrency (standard actor type, Abcl/1, Pool...) serialization of method acceptance ensures that there is at most one activation (implicit general mutual exclusion) which makes it safe but at the cost of restricting possible intra-object concurrency. See categories like Actalk-Synchro-Counters for a real full handling of activation synchronization and intra-object concurrency.'!


!EmptyBoundedBuffer methodsFor: 'script'!

put: item
	"This is an empty buffer, thus it defines and may accept a put:."

	contents addLast: item.
	self replace: (self isFull		"Compute replacement behavior."
					ifTrue: [FullBoundedBuffer new contents: contents maxSize: maxSize]
					ifFalse: [PartialBoundedBuffer new contents: contents maxSize: maxSize])! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

EmptyBoundedBuffer class
	instanceVariableNames: ''!


!EmptyBoundedBuffer class methodsFor: 'instance creation'!

new: maxSize
	"Create a new empty bounded buffer behavior.
	If maxSize is null, create a null bounded buffer behavior."

	^(maxSize > 0
		ifFalse: [NullBoundedBuffer]		"Bounded buffer of null size."
		ifTrue: [self]) new
			contents: (BoundedBufferObject new maxSize: maxSize)
			maxSize: maxSize! !

!EmptyBoundedBuffer class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGet: numberItems replyTo: Print! !

EmptyBoundedBuffer subclass: #PartialBoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Actor-Examples'!
PartialBoundedBuffer comment:
'Class PartialBoundedBuffer represents a bounded buffer not empty nor full.
It is defined as a subclass of class EmptyBoundedBuffer.
It only defines method getAndReplyTo:.
Note that if multiple inheritance would exist, class PartialBoundedBuffer should be defined as a subclass of both EmptyBoundedBuffer and FullBoundedBuffer. Here we have to duplicate method getAndReplyTo: from class FullBoundedBuffer.'!


!PartialBoundedBuffer methodsFor: 'script'!

getAndReplyTo: r
	"This is a partial buffer, thus it defines and may accept a get."

	r reply: contents removeFirst.
	self replace: (self isEmpty		"Compute replacement behavior."
					ifTrue: [EmptyBoundedBuffer new contents: contents maxSize: maxSize]
					ifFalse: [PartialBoundedBuffer new contents: contents maxSize: maxSize])! !
