ActorObject subclass: #OverdraftManager
	instanceVariableNames: 'account balance savings r '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Actor-Ext-Ex'!
OverdraftManager comment:
'Class OverdraftManager implements overdraft management of a ProtectedBankAccount.
It will compute and send/assign the replacement behavior of the protected bank account.

Instance Variables:

	account		<ProtectedBankAccount address>	is insensitive during the overdraft management.
	balance		<Integer>						the current balance of the ProtectedBankAccount.
	savings		<BankAccount address>			the savings account.
	r			<Address>						the reply destination to which the receipt
												acknowledgement has to be sent.'!


!OverdraftManager methodsFor: 'initialize'!

account: aPBankAccount balance: anInteger savings: aBankAccount r: aReplyDestination
	account := aPBankAccount.
	balance := anInteger.
	savings := aBankAccount.
	r := aReplyDestination! !

!OverdraftManager methodsFor: 'script'!

reply: receipt
	"Overdraft management."

	(Delay forMilliseconds: 5) wait.		"Simulation of network access latency!!"
	receipt = #withdrawalReceipt	
		ifTrue:
			["If the savings covers the overdraft, then the protected bank account new balance becomes 0."
			account replace:
				(ProtectedBankAccount new balance: 0 savings: savings).
			r reply: #overdraftCoveredReceipt]
		ifFalse:
			["Otherwise, we leave it as it was."
			account replace:	
				(ProtectedBankAccount new balance: balance savings: savings).
			r reply: #overdraftNotCoveredReceipt]! !

BankAccount subclass: #ProtectedBankAccount
	instanceVariableNames: 'savings '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Actor-Ext-Ex'!
ProtectedBankAccount comment:
'Class ProtectedBankAccount implements a bank account with provision and overdraft management.
It is a subclass of class BankAccount.
In order to manage overdrafts, a protected bank account will own some private bank account, called its savings, as an overdraft protection.
In case of overdraft, the protected bank account will inquire its savings for protection, by sending to it a request for withdrawal of the overdraft amount.
If the overdraft could be covered by the savings account, then the overdraft amount is withdrawn from the savings and used to rebalance the protected bank account (which then becomes null).
If the overdraft is too important for the savings, then previous behavior is unchanged (and a justice decision is requested!!?!!).

The point is following: the protected bank account cannot decide its further balance (and replacement behavior) before getting the reply (acceptance or reject) from the savings account. Meanwhile it becomes insensitive. A specific continuation will be created to manage the acknowledgement (receipt) from the savings and to compute and send the behavior replacement accordingly. The corresponding class is named OverdraftManager.

Instance Variables:

	savings		<aBankAccount address>		the savings account.'!


!ProtectedBankAccount methodsFor: 'initialize'!

balance: anInteger savings: aBankAccount
	balance := anInteger.
	savings := aBankAccount! !

!ProtectedBankAccount methodsFor: 'script'!

deposit: amount replyTo: r
	self replace: (self class new balance: balance + amount savings: savings).
	r reply: #depositReceipt!

withdraw: amount replyTo: r
	"In case of overdraft, become insensitive while waiting for management
	(the savings account is inquired for possible covering of the overdraft)."

	amount < balance
		ifTrue:
			["Safe case."
			self replace:
				(self class new balance: balance - amount savings: savings).
			r reply: #withdrawalReceipt]
		ifFalse:
			["Overdraft."
			savings withdraw: (amount - balance)	"Withdrawal request is sent to the savings account"
					replyTo:						"with an overdraft manager as continuation."
						(OverdraftManager new		
								account: aself	"Ths continuation stores all necessary info."
								balance: balance
								savings: savings
								r: r) active]

	"Note that there is no behavior replacement in case of overdraft.
	The protected bank account actor becomes insensitive until reception of
	a behavior replacement message conveying the replacement behavior computed
	and issued by the overdraft manager continuation.
	It does not reply a receipt either, the overdraft manager will."! !

!ProtectedBankAccount methodsFor: 'default classes'!

addressClass
	"We use external behavior replacement message for this example."

	^ExternalReplaceActorAddress! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ProtectedBankAccount class
	instanceVariableNames: ''!


!ProtectedBankAccount class methodsFor: 'instance creation'!

newBalance: balance savingsAmount: savingsAmount
	"Create a bank account and its savings account."

	^self new
		balance: balance
		savings: (BankAccount new balance: savingsAmount) active! !

!ProtectedBankAccount class methodsFor: 'example'!

example
	"self example"

	(self newBalance: 200 savingsAmount: 100) active
		deposit: 150 replyTo: Print;
		withdraw: 400 replyTo: Print;		"Create an overdraft (50)."
		balanceAndReplyTo: Print;
		deposit: 100 replyTo: Print;
		balanceAndReplyTo: Print! !

EmptyBoundedBuffer subclass: #EmptyBoundedBuffer2
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Actor-Ext-Ex'!

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

EmptyBoundedBuffer2 class
	instanceVariableNames: 'savedMethodSource '!


!EmptyBoundedBuffer2 class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	"self switchToEnabled: true"	"to switch to enable selectors activity."
	"self switchToEnabled: false"	"to switch back to EmptyBoundedBuffer naive definition."

	super exampleSize: maxSize numberItems: numberItems speedRatio: ratio!

switchToEnabled: boolean
	"Switch to optimized version with enabled selectors activity.
	May also revert to previous naive version (with message resend).
	Thanks to this method, we don't have to redefine all bounded buffer example classes."

	"self switchToEnabled: true"
	"self switchToEnabled: false"

	boolean
		ifTrue:
			[savedMethodSource := NullBoundedBuffer sourceMethodAt: #doesNotUnderstand:.
			NullBoundedBuffer
				removeSelector: #doesNotUnderstand:;
				compile:
'activityClass
	^EnabledSelectorsActorActivity'
				classified: #'default classes']
		ifFalse:
			[NullBoundedBuffer
				removeSelector: #activityClass;
				compile: savedMethodSource
				classified: #'message handling']! !
