Address subclass: #ExternalReplaceActorAddress
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Actor-Ext'!
ExternalReplaceActorAddress comment:
'Class ExternalReplaceActorAddress manages behavior replacement message.
This allows a replacement behavior to be computed externally and sent to the actor.
This is useful when the replacement behavior needs some non local information to be computed and therefore has to be computed externally by another actor.
Meanwhile the actor becomes insensitive (because it cannot accept any message with no behavior) while waiting (passively) for its replacement behavior.

Class ExternalReplaceActorAddress is a subclass of address kernel class Address.
Method replace: directly handles behavior replacement messages at the address level (and not at the behavior level). This is necessary because while insensitive the actor may not accept any message!!

Example of use is the management of overdraft. See class ProtectedBankAccount in category Actalk-Ext-Actor-Ext-Ex.
Also immediate extensions are implementations of futures, see class FutureActor.'!


!ExternalReplaceActorAddress methodsFor: 'behavior replacement'!

replace: replacementBehavior
	"Specify behavior replacement (assignment of a new active object and activity to the address).
	Note that this replacement behavior message is processed by the address
	and consequently may be sent externally by another actor.
	(See example of class OverdraftManager in category Actalk-Ext-Actor-Ext-Ex.)"

	self startNewActivity:
		(replacementBehavior activityClass new
			activeObject: replacementBehavior)! !

SingleMessageActivity subclass: #EnabledSelectorsActorActivity
	instanceVariableNames: 'enabledSelectors '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Actor-Ext'!
EnabledSelectorsActorActivity comment:
'Class EnabledSelectorActorActivity is a subclass of class SingleMessageActivity and provides control over acceptance of messages.
(It is actually also a virtual subclass of class EnabledSelectorsActivity but single inheritance restricts reuse from that class.)

Basically it looks only for messages that it knows (selectors of defined methods). This is useful to disable method acceptance (without relying on trapping errors and resending messages as in class NullBoundedBuffer).
Method nextMessage is redefined to look for messages that the class knows (defines or inherits).
See class EmptyBoundedBuffer2 in category Actalk-Ext-Actor-Ext-Ex as the resulting improved version of the bounded buffer example.'!


!EnabledSelectorsActorActivity methodsFor: 'initialize'!

privateInitialize
	"Initialize the collection of enabled selectors as the set of all script selectors."

	super privateInitialize.
	enabledSelectors := bself class allScriptSelectors! !

!EnabledSelectorsActorActivity methodsFor: 'message handling'!

nextMessage
	"Look into the mailbox for the first message matching one of the enabled selectors."

	^self mailBox firstMessageWithCondition: [:message |
		enabledSelectors includes: message selector]! !

ExternalReplaceActorAddress subclass: #FutureActor
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Actor-Ext'!
FutureActor comment:
'Class FutureActor implements futures (that is objects/actors whose behavior/value has not been assigned yet) simply as a newly created actor with no initial behavior.
In other words a FutureActor is simply a born insensitive actor (!!).
It will buffer all messages until its real nature (behavior) gets computed (externally and assigned through some behavior replacement message).

Actually there is no need for any implementation. No instance variable nor method is defined!!
Its superclass ExternalReplaceActorAddress could be used as well. But we define class FutureActor as a way to clearly identify the use of futures.

Note that the limitation of this model of future is that its value when computed may only be defined through a behavior.
Because Actalk is not fully uniform (not everything is an actor), basic entities such as numbers cannot be easily defined through behaviors. A future actor cannot become such a primitive object.
Future actors are useful to implement external actors, that is, in the terminology of Gul Agha, actors which provide interfacing with other program modules. One may provide the actual connection at execution time, meanwhile future actors will buffer all incoming messages.

One alternative to this implementation of future objects relies on more traditional objects and assignment, as well as semaphores. This could be found in category Actalk-FutureObject and will be used to implement eager type (also called future type) message passing.'!

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

FutureActor class
	instanceVariableNames: ''!


!FutureActor class methodsFor: 'example'!

example
	"self example"

	"This is an example of future actor which receives several transactions before actual specification of its initial behavior,
	computed externally and sent by a replacement message.
	Then it will be able to accept and serve the requests."

	self new
		deposit: 300 replyTo: Print;
		withdraw: 400 replyTo: Print;
		balanceAndReplyTo: Print;
		replace: (BankAccount new balance: 200)! !

