ObjectBodyObject subclass: #ObjectBodyVirtualBoundedBuffer
	instanceVariableNames: 'element '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Body-Examples'!
ObjectBodyVirtualBoundedBuffer comment:
'Class ObjectBodyVirtualBoundedBuffer implements the bounded buffer example by using the mailbox of the active object as the buffer itself.
The basic idea is to keep put: requests pending (in the mailbox) until they get served by a get request.
Instance variable element is a (primitive) means of communication between put: method and get method.

See also class AbclVirtualBoundedBuffer for a similar flavor but for a different activity model.'!


!ObjectBodyVirtualBoundedBuffer methodsFor: 'body'!

body
	"Body: keeps accepting/answering messages get.
	Every message put: is kept in the active object mailbox,
	until explicitly accepted by method get.
	The bounded buffer is the maibox itself !!"

	[true] whileTrue:
		[self answer:#(get)]! !

!ObjectBodyVirtualBoundedBuffer methodsFor: 'script'!

get
	"A get looks for a matching put: still pending."

	self answer: #(put:).
	"With the assumption that method put: will assign variable element
	as a (primitive) means of communication between put: and get."
	^element!

put: item
	"Saves item into communication variable element."

	element := item! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ObjectBodyVirtualBoundedBuffer class
	instanceVariableNames: ''!


!ObjectBodyVirtualBoundedBuffer class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := self new active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGet: numberItems! !

Get2BoundedBuffer subclass: #BodyGet2BoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Body-Examples'!
BodyGet2BoundedBuffer comment:
'This class implements an example of inheritance anomaly.'!


!BodyGet2BoundedBuffer methodsFor: 'state predicates'!

isOne
	^contents size = 1! !

!BodyGet2BoundedBuffer methodsFor: 'default classes'!

activityClass
	"Synchronization is specified by the activity body."

	^BodyGet2BoundedBufferActivity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

BodyGet2BoundedBuffer class
	instanceVariableNames: ''!


!BodyGet2BoundedBuffer class methodsFor: 'example'!

exampleGet2Size: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleGet2Size: 3 numberItems: 8 speedRatio: 3"
	"self exampleGet2Size: 3 numberItems: 8 speedRatio: 1/3"

	super exampleGet2Size: maxSize numberItems: numberItems speedRatio: ratio! !

ExplicitAcceptActivity subclass: #BodyBoundedBufferActivity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Body-Examples'!
BodyBoundedBufferActivity comment:
'This activity class specifies the synchronization of its associated behavior class.'!


!BodyBoundedBufferActivity methodsFor: 'body'!

body
	"Body: change/dispatch acceptance of messages according to the state of the buffer."
	"Note that calls to state predicates must be sent to the pseudo variable bself (behavior itself) reference to the behavior."

	[true] whileTrue:
		[self answer:
			(bself isEmpty
				ifTrue: [#(put:)]
				ifFalse: [bself isFull
							ifTrue: [#(get)]
							ifFalse: [#(get put:)]])]! !

BodyBoundedBufferActivity subclass: #BodyGet2BoundedBufferActivity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Body-Examples'!
BodyGet2BoundedBufferActivity comment:
'This activity class specifies the synchronization of its associated behavior class.'!


!BodyGet2BoundedBufferActivity methodsFor: 'body'!

body
	"Body: change/dispatch acceptance of messages according to the state of the buffer."

	[true] whileTrue:
		[self answer:
			(bself isEmpty
				ifTrue: [#(put:)]
				ifFalse: [bself isOne
							ifTrue: [#(get put:)]
							ifFalse: [bself isFull
										ifTrue: [#(get2 get)]
										ifFalse: [#(get2 get put:)]]])]! !

BodyBoundedBufferActivity subclass: #BodyGgetBoundedBufferActivity
	instanceVariableNames: 'afterPut '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Body-Examples'!
BodyGgetBoundedBufferActivity comment:
'This activity class specifies the synchronization of its associated behavior class.'!


!BodyGgetBoundedBufferActivity methodsFor: 'body'!

body
	"Body: change/dispatch acceptance of messages according to the state of the buffer."

	[true] whileTrue:
		[self answer:
			(bself isEmpty
				ifTrue: [#(put:)]
				ifFalse: [bself isFull
							ifTrue: [#(get)]
							ifFalse: [afterPut
										ifTrue: [#(get put:)]	
	"Note: gget should better be stated before get because its chances to be served are smaller."
										ifFalse: [#(gget get put:)]]])]! !

!BodyGgetBoundedBufferActivity methodsFor: 'events'!

kernelEventComplete: aMessage
	"Record if message being completed is a put:."

	super kernelEventComplete: aMessage.
	afterPut := aMessage selector = #put:! !

SimulatedBodyObject subclass: #SimulatedBodyBoundedBuffer
	instanceVariableNames: 'contents maxSize '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Body-Examples'!
SimulatedBodyBoundedBuffer comment:
'Class SimulatedBodyBoundedBuffer redefines the bounded buffer example with a simulated body.'!


!SimulatedBodyBoundedBuffer methodsFor: 'initialize'!

initialize: anInteger
	maxSize := anInteger.
	contents := BoundedBufferObject new maxSize: maxSize! !

!SimulatedBodyBoundedBuffer methodsFor: 'script'!

body
	"Body: change/dispatch acceptance of messages according to the state of the buffer."

	[true] whileTrue:
		[self answer:
			(self isEmpty
				ifTrue: [#(put:)]
				ifFalse: [self isFull
							ifTrue: [#(get)]
							ifFalse: [#(get put:)]])]!

get
	^contents removeFirst!

put: item
	contents addLast: item! !

!SimulatedBodyBoundedBuffer methodsFor: 'state predicates'!

isEmpty
	^contents isEmpty!

isFull
	^contents isFull! !

!SimulatedBodyBoundedBuffer methodsFor: 'printing'!

printOn: aStream
	"Print as default followed by its contents."

	super printOn: aStream.
	contents printOn: aStream! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SimulatedBodyBoundedBuffer class
	instanceVariableNames: ''!


!SimulatedBodyBoundedBuffer class methodsFor: 'instance creation'!

new: size
	^self new initialize: size! !

!SimulatedBodyBoundedBuffer class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGet: numberItems! !

ObjectBodyObject subclass: #Thief
	instanceVariableNames: 'name bag organization '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Body-Examples'!
Thief comment:
'Class Thief implements a thief whose activity is to rob (as well as to be robbed by) other thieves, that is fetching one of its things. If a thief has nothing to reply then he kills himself (replying his life).
The game is over when only one thief remains!!

Class Thief is defined as a subclass of behavior class ObjectBodyObject in order to be able to directly specify its body.

The body explicitly specifies when and which messages it is willing to serve, e.g., to be robbed or to receive a thing he requested.
Note that in order to increase asynchrony, fetching one thing is expressed with two asynchronous unidirectional message sends (requesting and receiving).
This is also one simple way to manage the issue of thieves which happen to be killed and which then cannot reply anymore.

Instance Variables:

	name		<String>						the name of the thief.
	bag			<OrderedCollection>			the set of things owned by the thief.
	organization	<ThiefOrganization address>	the organization to which the thief belongs (is registered).'!


!Thief methodsFor: 'initialize'!

name: aString owns: arrayOfThings organization: aThiefOrganization
	name := aString.
	bag := arrayOfThings asOrderedCollection.
	organization := aThiefOrganization! !

!Thief methodsFor: 'body'!

body
	"Main activity for a thief."

	| victim |
	[true] whileTrue:
		["First look if something requested has arrived."
		(self unBlockingAnswer: #(reply:))
			ifFalse:
				["If not, get angry and look for a victim."
				victim := (organization oneExcept: aself) value.
				victim isNil
					ifTrue:		
						["If no victim can be found, this means that the thief is the only remaining."
						Transcript show: self printString , ' is a total winner!!'; cr.
						Transcript show: self printString , ' owns ' , bag printString; cr.
						^'exit from the body loop']
					ifFalse:
						["Otherwise, it orders its victim to give him something."
						victim giveSomethingAndReplyTo: aself]].
		"After this satisfaction, sleep for a while."
		(Delay forMilliseconds: (Random new next * 1000) rounded) wait.
		"When awaking the thief is vulnerable to other thieves requests."
		self unBlockingAnswer: #(giveSomethingAndReplyTo:).
		"Trace the amount of things owned, and loop again."
		Transcript show: name , '(' , bag size printString , ') ']! !

!Thief methodsFor: 'script'!

giveSomethingAndReplyTo: r
	bag isEmpty
		ifTrue:
			["If no more thing to give, then the thief kills itself (and replies his own life...)."
			organization unregister: aself.
			Transcript show: self printString , ' is killed by ' , r printString; cr.
			self terminate]
		ifFalse:
			["Otherwise, it replies one thing among the ones it owns."
			r reply: bag removeOneRandom]!

reply: aThing
	bag add: aThing! !

!Thief methodsFor: 'printing'!

printOn: aStream
	aStream nextPutAll: name! !

IRBoundedBuffer subclass: #BodyBoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Body-Examples'!
BodyBoundedBuffer comment:
'Class BodyBoundedBuffer implements the bounded buffer example with use of a body to specify synchronization.
The body specifies which messages it may accept according to its state.
Note that synchronization code is centralized within the body. As a result this model is not very good at reusing specifications because it has to fully redefine the body.

Further note that there is an alternative version of the bounder buffer which uses its maibox as the buffer. See class ObjectBodyVirtualBoundedBuffer (and also AbclVirtualBoundedBuffer).'!


!BodyBoundedBuffer methodsFor: 'default classes'!

activityClass
	"Synchronization is specified by the activity body."

	^BodyBoundedBufferActivity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

BodyBoundedBuffer class
	instanceVariableNames: ''!


!BodyBoundedBuffer class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	super exampleSize: maxSize numberItems: numberItems speedRatio: ratio! !

BodyBoundedBuffer subclass: #ObjectBodyBoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Body-Examples'!
ObjectBodyBoundedBuffer comment:
'Class ObjectBodyBoundedBuffer redefines the example of the bounded buffer with the body method directly specified by the behavior itself.
On one hand, there is no need to call oself for accessing behavior predicates or data, but on the other hand there is no more complete separation between behavior and synchronization.'!


!ObjectBodyBoundedBuffer methodsFor: 'body'!

body
	"Body: change/dispatch acceptance of messages according to the state of the buffer."

	[true] whileTrue:
		[self answer:
			(self isEmpty
				ifTrue: [#(put:)]
				ifFalse: [self isFull
							ifTrue: [#(get)]
							ifFalse: [#(get put:)]])]! !

!ObjectBodyBoundedBuffer methodsFor: 'default classes'!

activityClass
	"In order to delegate the body from the activity to the behavior."

	^ObjectBodyActivity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ObjectBodyBoundedBuffer class
	instanceVariableNames: ''!


!ObjectBodyBoundedBuffer class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	super exampleSize: maxSize numberItems: numberItems speedRatio: ratio! !

ImplicitReplyObject subclass: #ThiefOrganization
	instanceVariableNames: 'thieves '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Body-Examples'!
ThiefOrganization comment:
'Class ThiefOrganization implements an organization of thieves where they can register (at start), unregister (when killed), and ask for a victim (one of the other thieves).

Instance Variables:

	thieves		<OrderedCollection>		all registered thieves.

See comment of class Thief.'!


!ThiefOrganization methodsFor: 'initialize'!

initialize
	thieves := OrderedCollection new! !

!ThiefOrganization methodsFor: 'script'!

oneExcept: aThief
	"Return a thief (victim) requested.
	It cannot be the requester.
	Return nil if only one thief is remaining."

	^thieves size > 1
		ifTrue: [thieves oneRandomExcept: aThief]
		ifFalse: [nil]!

register: aThief
	thieves add: aThief!

unregister: aThief
	thieves remove: aThief
		ifAbsent: [Transcript show: aThief printString , ' : you live only twice!!']! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ThiefOrganization class
	instanceVariableNames: ''!


!ThiefOrganization class methodsFor: 'instance creation'!

new
	^super new initialize! !

!ThiefOrganization class methodsFor: 'example'!

example
	"self example"

	"Thief cleanUp"	"to stop before the winner takes it all if you get tired!!"

	| mafia |
	mafia := self new active.
	mafia
		register: (Thief new name: 'Jeeps' owns: #(PowerBook bass bicycle stereo) organization: mafia) active;
		register: (Thief new name: 'Jeff' owns: #(scuba PC bike Tshirt) organization: mafia) active;
		register: (Thief new name: 'Laurent' owns: #(coffeeMachine kanjiBook camera Self) organization: mafia) active;
		register: (Thief new name: 'Jacques' owns: #(KoreanDictionary lambda curry garlic) organization: mafia) active;
		register: (Thief new name: 'Gao' owns: #(alphaBeta transputer neckTie glasses) organization: mafia) active! !

GgetBoundedBuffer subclass: #BodyGgetBoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Body-Examples'!
BodyGgetBoundedBuffer comment:
'This class implements an example of inheritance anomaly.'!


!BodyGgetBoundedBuffer methodsFor: 'default classes'!

activityClass
	"Synchronization is specified by the activity body."

	^BodyGgetBoundedBufferActivity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

BodyGgetBoundedBuffer class
	instanceVariableNames: ''!


!BodyGgetBoundedBuffer class methodsFor: 'example'!

exampleGgetSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleGgetSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleGgetSize: 3 numberItems: 8 speedRatio: 1/3"

	"Note that there is no guarantee that all gget requests will be served.
	As a result the bounded buffer may stop before all producer and consumer requests had been served."

	super exampleGgetSize: maxSize numberItems: numberItems speedRatio: ratio! !
