Address subclass: #GenericSendAddress
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-GenericSend'!
GenericSendAddress comment:
'Class GenericSendAddress is an abstract class which parameterizes structure of the message and dispatching of message types and modes.
It is defined as a subclass of address class Address.

Basic and default structure of the message is as an instance of standard Smalltalk-80 class Message. In order to insert new information (sender, reply destination, arrival time...) which may be useful in specific subclasses, the message structure is made generic, by calling parameter method invocationClassFor: of the associated activity class.

The class also provides generic dispatch of the message received along possible message send types and modes.
As an example, ABCL/1 three types of message passing (asynchronous, synchronous, and future), plus the express mode are easily defined as a subclass (classes Abcl1Address and Abcl3Address).
Also implicit reply (class ImplicitReplyAddress) and POOL synchronous message passing and public routines (class PoolAddress) are easily defined as subclasses.

Parameter method receiveMessage: is redefined in order to generify message structure and message send.

New parameter methods are:

	receiveGenericMessage:								to dispatch the generified message,

	sendTypeOf:											to find the type of the message send, e.g., asynchronous, synchronous...

	sendModeOf:										to find the mode of the message send, that is on which message queue it should be enqueued,

	self receiveMessage:withSendType:withSendMode:		to first dispatch according to the mode,

	self receiveMessage:withSendType:inMessageQueue:	to further dispatch according to the type,

	insertReplyDestination:intoMessage:					to express how to insert a future object as a reply destination within a future type message.'!


!GenericSendAddress methodsFor: 'message passing'!

receiveGenericMessage: aMessage
	"Receive a generic invocation/message."
	"Generic dispatch along the message send type and mode."

	^self receiveMessage: aMessage
		withSendType: (self sendTypeOf: aMessage)
		withSendMode: (self sendModeOf: aMessage)!

receiveMessage: aMessage
	"Receive the message and make it generic (that is possibly add some more information slots)."
	"Parameter method invocationClassFor: is defined in activity classes."
	"Note that as an optimization, we check if the generic invocation class is class Message
	in order to create a new invocation only if needed (if some more slots are defined)."

	| invocationClass |
	invocationClass := activity invocationClassFor: aMessage.
	^self receiveGenericMessage:
		(invocationClass = Message
			ifTrue:
				[aMessage]
			ifFalse:
				[invocationClass new
					setSelector: aMessage selector
					arguments: aMessage arguments])! !

!GenericSendAddress methodsFor: 'message dispatching'!

receiveMessage: aMessage withSendType: type inMessageQueue: queue
	"Generic dispatch on the message send type."
	"This is a parameter method."
	"Default recognizes only asynchronous type."

	type = #asynchronous
		ifTrue: [^self asynchronousSend: aMessage inMessageQueue: queue].
	^self error: 'unrecognized message type: ' , type printString!

receiveMessage: aMessage withSendType: type withSendMode: mode
	"Generic dispatch on the message send mode."
	"This is a parameter method."
	"Default recognizes only standard mode."

	mode = #standard
		ifTrue: [^self receiveMessage: aMessage withSendType: type inMessageQueue: mailBox].
	self error: 'unrecognized message mode: ' , mode printString!

sendModeOf: aMessage
	"Check and return the message send mode."
	"This is a parameter method."
	"Default is standard mode."

	^#standard!

sendTypeOf: aMessage
	"Check and return the message send type."
	"This is a parameter method."
	"Default is asynchronous type."

	^#asynchronous! !

!GenericSendAddress methodsFor: 'message passing types'!

futureSend: aMessage withFutureClass: futureClass inMessageQueue: queue
	"Implement future type message passing as an asynchronous message send
	with the future object inserted within the message.
	Immediately return the future object."

	| aFuture |
	aFuture := futureClass new.
	self asynchronousSend:
			(self insertReplyDestination: aFuture intoMessage: aMessage)
		inMessageQueue: queue.
	^aFuture!

insertReplyDestination: r intoMessage: aMessage
	"Insert a reply destination into a message and return the extended message."
	"This is a parameter method."
	"No default. It should be defined in subclasses."

	^self subclassResponsibility!

synchronousSend: aMessage inMessageQueue: queue
	"Implement synchronous message passing as waiting onto a future type message passing."

	^(self futureSend: aMessage withFutureClass: SAFuture inMessageQueue: queue) value! !

Activity subclass: #CurrentMessageActivity
	instanceVariableNames: 'currentMessage '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-GenericSend'!
CurrentMessageActivity comment:
'Class CurrentMessageActivity provides a pseudo-variable named currentMessage which references the current message being performed.

Note that currentMessage denotates current message all along acceptance BECAUSE we may assume there is one message at a time.
This simple implementation is not valid in an intra-object concurrency context.
We could then rely on a dictionary (indexed by current process) strategy as the one used to manage post-actions in the class ConcurrentSmalltalkActivity, in category Actalk-Ext-Suspend.

Class CurrentMessageActivity is useful to define access to the sender (of current message), see subclass WithSenderActivity.
Pseudo-variable currentMessage together with generic events also provide most of OCore meta-level archirecture needed to express some synchronization onto acceptance of messages (actually by aborting them and reaccepting them eventually), see category Actalk-Ext-OCore-Ex.'!


!CurrentMessageActivity methodsFor: 'accessing'!

currentMessage
	"Return current message."

	^currentMessage! !

!CurrentMessageActivity methodsFor: 'events'!

kernelEventAccept: aMessage
	"Record currentMessage."
	"Note: this implementation is safe because we assume serialization. See class comment."

	super kernelEventAccept: aMessage.
	currentMessage := aMessage! !
