Message subclass: #WithReplyDestInvocation
	instanceVariableNames: 'replyDest '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-ImplicitReply'!
WithReplyDestInvocation comment:
'Class ReplyDestInvocation is a subclass of class Message.
It adds an extra instance variable to implicitly contain and convey the reply destination (a future object). This class is used for implicit reply active objects.'!


!WithReplyDestInvocation methodsFor: 'accessing'!

replyDest
	^replyDest!

replyDest: replyDestination
	replyDest := replyDestination! !

ActiveObject subclass: #ImplicitReplyObject
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-ImplicitReply'!
ImplicitReplyObject comment:
'Only purpose is to dispatch onto classes ImplicitReplyActivity/Address.
Please see class ImplicitReplyAddress comment.'!


!ImplicitReplyObject methodsFor: 'default classes'!

activityClass
	"This active object behavior class only purpose is to dispatch onto that activity class."

	^ImplicitReplyActivity! !

ImplicitReplyObject subclass: #OptimizedImplicitReplyObject
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-ImplicitReply'!
OptimizedImplicitReplyObject comment:
'Only purpose is to dispatch onto classes OptimizedImplicitReplyActivity/Address.
Please see class OptimizedImplicitReplyAddress comment.'!


!OptimizedImplicitReplyObject methodsFor: 'default classes'!

activityClass
	"This active object behavior class only purpose is to dispatch onto that activity class."

	^OptimizedImplicitReplyActivity! !

Activity subclass: #ImplicitReplyActivity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-ImplicitReply'!
ImplicitReplyActivity comment:
'Please see class ImplicitReplyAddress comment.'!


!ImplicitReplyActivity methodsFor: 'message handling'!

performMessage: anInvocation
	"Perform the message and assign (reply) the returned value to the future object (implicit reply destination)."

	anInvocation replyDest reply: (super performMessage: anInvocation)! !

!ImplicitReplyActivity methodsFor: 'default classes'!

addressClass
	"ImplicitReplyActivity method
		performMessage:
			makes assumption that the message has been extended
			(into a ReplyDestMessage conveying a future object implicit reply destination) by
	ImplicitReplyAddress method
		insertReplyDestination:intoMessage:"

	^ImplicitReplyAddress!

invocationClassFor: aMessage
	"Invocation with one slot to record the reply destination of the message."

	^WithReplyDestInvocation! !

!ImplicitReplyActivity methodsFor: 'compatibility constraints'!

addressConstraint
	"Enforce assumption of method addressClass"

	^ImplicitReplyAddress! !

ImplicitReplyActivity subclass: #OptimizedImplicitReplyActivity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-ImplicitReply'!
OptimizedImplicitReplyActivity comment:
'Please see class OptimizedImplicitReplyAddress comment.'!


!OptimizedImplicitReplyActivity methodsFor: 'message handling'!

performMessage: aMessage
	"Bypass insertion of the reply destination if this is an asynchronous type of message send."

	(address sendTypeOf: aMessage) = #asynchronous
		ifTrue: [bself performMessage: aMessage]
		ifFalse: [super performMessage: aMessage]! !

!OptimizedImplicitReplyActivity methodsFor: 'default classes'!

addressClass
	"OptimizedImplicitReplyActivity method
		performMessage:
			makes assumption that the message has been extended except for asynchronous message send (selector starting with an 'A') by
	OptimizedImplicitReplyAddress method
		insertReplyDestination:intoMessage:"

	^OptimizedImplicitReplyAddress!

invocationClassFor: aMessage
	"Invocation with one slot to record the reply destination of the message
	except for asynchronous message send."

	^(address sendTypeOf: aMessage) = #asynchronous
		ifTrue: [Message]
		ifFalse: [WithReplyDestInvocation]! !

!OptimizedImplicitReplyActivity methodsFor: 'compatibility constraints'!

addressConstraint
	"Enforce assumption of method addressClass"

	^OptimizedImplicitReplyAddress! !

GenericSendAddress subclass: #ImplicitReplyAddress
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-ImplicitReply'!
ImplicitReplyAddress comment:
'Classes ImplicitReplyActivity and ImplicitReplyAddress implement a better integration of message passing between active objects within standard Smalltalk-80 framework.
Standard Smalltalk way of writing and using methods is preserved.
Implicit reply and returned value is specified as usual by using the ^.
(Note that if no ^ is specified, self is returned as a value as usual. In order to return the reference to the active object itself, one should explicitly state ^aself).

Sending a message to an ImplicitReplyAddress returns a future object (see category Actalk-Future) which will eventually contain the value returned by the method performed. The sender (client) may make use of the value by explicitly stating the selector value to wait for the value of the future object. It may also discard the future object if a simple asynchronous message send is appropriate.'!


!ImplicitReplyAddress methodsFor: 'message dispatching'!

receiveMessage: aMessage withSendType: type inMessageQueue: queue
	"Recognize future type."

	type = #future
		ifTrue: [^self futureSend: aMessage withFutureClass: SAFuture inMessageQueue: queue].
	^super receiveMessage: aMessage withSendType: type inMessageQueue: queue!

sendTypeOf: aMessage
	"Assume that every message sent should create a future to hold the reply.
	The future is returned (as for a future type message passing)."

	^#future! !

!ImplicitReplyAddress methodsFor: 'message passing types'!

insertReplyDestination: r intoMessage: aMessage
	"Insert the reply destination into the invocation.
	Return the message."

	^aMessage replyDest: r! !

!ImplicitReplyAddress methodsFor: 'compatibility constraints'!

activityConstraint
	"ImplicitReplyAddress way of receiving a future type message
	makes assumption that the future object will eventually receive its value computed by
	ImplicitReplyActivity method
		performMessage:"

	^ImplicitReplyActivity! !

ImplicitReplyAddress subclass: #OptimizedImplicitReplyAddress
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-ImplicitReply'!
OptimizedImplicitReplyAddress comment:
'Classes OptimizedImplicitReplyActivity and OptimizedImplicitReplyAddress implement a slightly optimized version of implicit reply.
The point is following: many transmissions are asynchronous. In such a case creating a future object means a waste of resources.

A simple way of tagging asynchronous message passing is by using special selectors (starting with capital letter A). In that case, the message sending and computation proceeds as for kernel active objects.

Note that this means that a same selector may not be used both for asynchronous message passing and other types (future or/and synchronous). However this is rarely the case, except when the value is of no use but the sender may want to be able to synchronize on the acknowledgement. In such a (rare) case the method may be duplicated.'!


!OptimizedImplicitReplyAddress methodsFor: 'message dispatching'!

sendTypeOf: aMessage
	"Dispatch onto asynchronous or future type message passing,
	according to the first letter of the selector (A for asynchronous)."

	^(aMessage selector at: 1) = $A
		ifTrue: [#asynchronous]
		ifFalse: [#future]! !

!OptimizedImplicitReplyAddress methodsFor: 'compatibility constraints'!

activityConstraint
	"OptimizedImplicitReplyAddress way of receiving a future type message
	makes assumption that, except for asynchronous message sends,
	the future object will eventually receive its value computed by
	OptimizedImplicitReplyActivity method
		performMessage:"

	^OptimizedImplicitReplyActivity! !

