PoolObject subclass: #Searcher
	instanceVariableNames: 'info '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Pool-Examples'!
Searcher comment:
'Class Searcher implements the search for a key into a symbol table.
See class SymbolTable comment.'!


!Searcher methodsFor: 'body'!

body
	"Body: accept only one message to start up the search."

	self answer: #(goLookFor:inTable:)! !

!Searcher methodsFor: 'script'!

goLookFor: key inTable: table
	"Look for a key in a symbol table and wait for the reply."

	table lookFor: key forClient: aself.
	self answer: #(reply:).
	"The reply has assigned the result of the search into instance variable info."
	^info!

reply: theInfo
	"Receive the result of the search. Assign it to instance variable info."

	info := theInfo! !

PoolObject subclass: #PoolPhilosopher
	instanceVariableNames: 'number room leftFork rightFork numberSteps '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Pool-Examples'!
PoolPhilosopher comment:
'Class PoolPhilosopher (with classes PoolFork and PoolRoom) implements the seminal synchronization example of the dining philosophers.

This problem consists of n philosophers sitting around a table. N forks are placed between them. This means that a philosopher shares his left fork with his left neighbour (who sees it as his right fork) and his right fork with his right neighbour (who sees it as his left fork).
Each philosopher alternatively thinks and then eats. In order to eat a philosopher attempts at picking up his left fork and then his right fork. After eating he releases both forks.

There are two problems to be solved.
First is deadlock avoidance. Suppose that every philosopher picks up his left fork. Then they will all infinitely wait for their right forks to be free (because their right fork is held by the philosopher sitting at their right).
Second is starvation avoidance. One philosopher may starve by being unable to eat if his two forks are never both free.

This implementation follows the algorithm proposed in the POOL language where philosophers and forks are active objects.
The initial implementation in a previous version of Actalk was achieved by a group of students of DESS of University of Nantes, under the guidance of Jean Bezivin and Olivier Roux.

Note that a philosopher never answers any message, only the body is used.

Note that in order to avoid deadlock we need to introduce a dining room where philosophers have to enter in order to ensure that there is at most n-1 philosophers trying to eat.

Another solution without the need for a room needs prior check of availability of forks requested. Therefore synchronization depends on consultation of the arguments (which forks are requested). The answer: construct does not allow such inquiry.
Thus that solution will be described in a synchronization formalism using guards, see in category Actalk-Synchro-Guards-Ex.

Instance Variables:

	number			<Integer>			number of the philosopher (for tracing purposes).
	room			<PoolRoom address>	the room where they think.
										This is used to restrict access to forks (and food!!)
										to only n-1 philosophers.
	leftFork			<PoolFork address>	left fork of the philosopher.
	rightFork		<PoolFork address>	right fork of the philosopher.
	numberSteps	<Integer>			number of times the philosopher(s) will think and eat.'!


!PoolPhilosopher methodsFor: 'initialize'!

number: anInteger room: aRoom leftFork: fork1 rightFork: fork2 numberSteps: steps
	number := anInteger.
	room := aRoom.
	leftFork := fork1.
	rightFork := fork2.
	numberSteps := steps! !

!PoolPhilosopher methodsFor: 'body'!

body
	"Body: do numberSteps times his life steps, that is: think, enter the room to eat, and exit."
	"Note that this active object class has no script method as it receives no message."

	1 to: numberSteps do: [: i |
		self think.
		Transcript show: self printString , ' wants to eat for the ' , i printString , 'th time and therefore to enter the room.'; cr.
		room enter: aself.
		Transcript show: self printString , ' wants to pick up his left fork: ' , leftFork printString; cr.
		leftFork pickUpBy: aself.
		Transcript show: self printString , ' picked up his left fork: ' , leftFork printString; cr.
		Transcript show: self printString , ' wants to pick up his right fork: ' , rightFork printString; cr.
		rightFork pickUpBy: aself.
		Transcript show: self printString , ' picked up his right fork: ' , rightFork printString; cr.
		self eat.
		leftFork putDownBy: aself.
		Transcript show: self printString , ' has put down his left fork: ' , leftFork printString; cr.
		rightFork putDownBy: aself.
		Transcript show: self printString , ' has put down his right fork: ' , rightFork printString; cr.
		room exit: aself]! !

!PoolPhilosopher methodsFor: 'private routines'!

eat
	| delay |
	delay := (Random new next * 5) rounded + 1.
	Transcript show: self printString , ' eats with forks '
			, leftFork printString , ' and ' , rightFork printString
			, ' during ' , delay printString , ' seconds.'; cr.
	(Delay forSeconds: delay) wait!

think
	| delay |
	delay := (Random new next * 5) rounded + 1.
	Transcript show: self printString , ' thinks during ' , delay printString , ' seconds.'; cr.
	(Delay forSeconds: delay) wait! !

!PoolPhilosopher methodsFor: 'printing'!

printOn: aStream
	"Print as its class name followed by its number."

	aStream nextPutAll: self class name.
	number printOn: aStream! !

PoolObject subclass: #PoolRoom
	instanceVariableNames: 'n numberSteps occupancy philosophers forks '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Pool-Examples'!
PoolRoom comment:
'Class PoolRoom implements a dining room where philosophers enter in order to eat. This is to ensure that there is at most n-1 philosophers trying to eat to avoid deadlock.
Class PoolRoom is also used for initialization and starting of the simulation.

See also class PoolPhilosopher comment.'!


!PoolRoom methodsFor: 'initialize'!

initialize: anInteger numberSteps: steps
	n := anInteger.
	numberSteps := steps.
	occupancy := 0!

makeForks
	"Create and initialize the n forks."

	forks := Array new: n.
	1 to: n do: [:i |
		forks at: i put:
			(PoolFork new number: i) active]!

makePhilosophers
	"Create and initialize the n philosophers."

	philosophers := Array new: n.
	1 to: n do: [:i |
		philosophers at: i put:
			(PoolPhilosopher new
				number: i
				room: aself
				leftFork: (forks at: i)
				rightFork: (forks at: (i \\ n) + 1)
				numberSteps: numberSteps)
					active]! !

!PoolRoom methodsFor: 'body'!

body
	"Body: always accept exit, but accept enter only if room is not full (n -1 occupants)."

	"First message is for initializing and starting up the simulation."
	self answer: #(start).
	[true] whileTrue: 
		[self answer:
			(occupancy < (n - 1)
				ifTrue: [#(enter: exit:)]
				ifFalse: [#(exit:)])]! !

!PoolRoom methodsFor: 'script'!

enter: aPhilosopher
	Transcript show: aPhilosopher printString , ' enters ' , self printString; cr.
	occupancy := occupancy + 1!

exit: aPhilosopher
	Transcript show: aPhilosopher printString , ' exits ' , self printString; cr.
	occupancy := occupancy - 1!

start
	"Initialize and start the forks and philosophers active objects."
	"Note that this method must be a script method (and not a private initialization method)
	because the room active object must have been created prior to creation of philosophers."

	self makeForks.
	self makePhilosophers! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

PoolRoom class
	instanceVariableNames: ''!


!PoolRoom class methodsFor: 'example'!

example: n
	"self example: 5"

	(self new initialize: n numberSteps: 3) active start! !

PoolObject subclass: #PoolFork
	instanceVariableNames: 'number '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Pool-Examples'!
PoolFork comment:
'Class PoolFork implements forks in the dining philosophers problem as POOL active objects.

Instance Variables:

	number		<Integer>	number of the fork (for tracing purposes).'!


!PoolFork methodsFor: 'initialize'!

number: anInteger
	number := anInteger! !

!PoolFork methodsFor: 'body'!

body
	"Body: accept pickUp, then pickDown, etc..."

	[true] whileTrue: 
		[self answer: #(pickUpBy:).
		self answer: #(putDownBy:)]! !

!PoolFork methodsFor: 'script'!

pickUpBy: aPhilosopher
	"Do nothing. This is pure synchronization."!

putDownBy: aPhilosopher
	"Do nothing. This is pure synchronization."! !

!PoolFork methodsFor: 'printing'!

printOn: aStream
	"Print as its class name followed by its number."

	aStream nextPutAll: self class name.
	number printOn: aStream! !

PoolObject subclass: #SymbolTable
	instanceVariableNames: 'myKey myInfo left right '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Pool-Examples'!
SymbolTable comment:
'Class SymbolTable implements a symbol table implemented in a distributed fashion.
The initial algorithm is described in POOL by Pierre America in POOL-T: a Parallel Object-Oriented Language, in the OOCP book, MIT-Press, pages 213-218.
This algorithm is close to the one described in class QuickSortTree, in category Actalk-Ext-Accept-Examples.'!


!SymbolTable methodsFor: 'script'!

insert: key with: info
	"Insert a new association into the table.
	Note that acknowledgement is immediately returned (so that the client does not have to wait).
	All processing is done as post actions."

	self post:
		[myKey isNil
			ifTrue:				
				["If this node is still empty, then initialize the node with the key."
				myKey := key.
				myInfo := info.
				left := SymbolTable new active.
				right := SymbolTable new active]
			ifFalse:				
				["Otherwise, insert it according to order."
				key = myKey
					ifTrue:
						[myInfo := info]
					ifFalse:
						[key < myKey
							ifTrue: [left insert: key with: info]
							ifFalse: [right insert: key with: info]]]]!

lookFor: key forClient: client
	"Look for a key into the table.
	Note that acknowledgement is immediately returned (so that the client does not have to wait).
	All processing is done as post actions."

	self post:
		[myKey isNil								
			ifTrue:
				["Key not found, then return nil."
				client reply: nil]
			ifFalse:
				["Otherwise, we can compare both keys."
				key = myKey					
					ifTrue:
						["If same key, then return the associated info."
						client reply: myInfo]
					ifFalse:
						["Otherwise propagate the search to one of the sub nodes according to order."
						key < myKey
							ifTrue: [left lookFor: key forClient: client]
							ifFalse: [right lookFor: key forClient: client]]]]! !

!SymbolTable methodsFor: 'public routines'!

PublicSearchFor: key
	"Search for a key by creating a new searcher active object."
	"Note that method searchFor: MUST be a public routine
	because otherwise it creates a deadlock
	(because the searcher will issue a request to the symbol table).
	This is safe because this method makes no assignment."

	^Searcher new active goLookFor: key inTable: aself! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SymbolTable class
	instanceVariableNames: ''!


!SymbolTable class methodsFor: 'example'!

counterExample
	"self counterExample"

	^self new active
		insert: #b with: 2;
		insert: #d with: 4;
		insert: #a with: 1;
		insert: #c with: 3;
		PublicSearchFor: #z!

example
	"self example"

	^self new active
		insert: #b with: 2;
		insert: #d with: 4;
		insert: #a with: 1;
		insert: #c with: 3;
		PublicSearchFor: #c! !

