ImplicitReplyAddress subclass: #PoolAddress
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Pool'!
PoolAddress comment:
'Trio classes PoolObject, PoolActivity and PoolAddress implement POOL body model, explicit message acceptance, post actions, synchronous message passing, and (object-like) public routines.

Address class PoolAddress implements synchronous message passing as a standard and dispatches onto public routine handling.
Class PoolActor is defined as a subclass of class ImplicitReplyAddress (to handle implicit reply).
See class PoolActivity comment.'!


!PoolAddress methodsFor: 'message passing'!

receiveMessage: aMessage withSendType: type inMessageQueue: queue
	"Dispatch along public routine handling or synchronous message passing."

	type = #publicRoutine
		ifTrue: [^activity acceptPublicRoutineMessage: aMessage].
	type = #synchronous
		ifTrue: [^self synchronousSend: aMessage inMessageQueue: queue].
	^super receiveMessage: aMessage withSendType: type inMessageQueue: queue!

sendTypeOf: aMessage
	"A public routine message is tagged through the first letter of the selector (P for public).
	Otherwise, this is synchronous message passing."

	^(aMessage selector at: 1) = $P
		ifTrue: [#publicRoutine]
		ifFalse: [#synchronous]! !

!PoolAddress methodsFor: 'compatibility constraints'!

activityConstraint
	"PoolAddress method
		receiveMessage:withSendType:inMessageQueue:
			may call
	PoolActivity method
		acceptPublicRoutineMessage:"

	^PoolActivity! !

ObjectBodyActivity subclass: #PoolActivity
	instanceVariableNames: 'postBlock '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Pool'!
PoolActivity comment:
'Trio classes PoolObject, PoolActivity and PoolAddress implement POOL body model, explicit message acceptance, post actions, synchronous message passing, and (object-like) public routines.

See an article onto the POOL language by Pierre America in the OOCP book, pages 199-220.

Activity class PoolActivity is defined as a subclass of class ObjectBodyActivity, as body specification is delegated to the behavior.

It defines a default body which accepts any kind of message successively (as for standard kernel active objects).
It provides post actions which can be executed after returning the value, thus freeing the sender from unneeded waiting.
It provides public routines. Public routines offer object-like methods which are directly computed by the behavior.
They are useful to encapsulate public protocols. In fact they can be necessary to avoid deadlock as it is the case when there is recursion (possibly mutual). (Remember that all message passing in POOL is synchronous thus recursion leads to deadlock).
See for instance the example of class SymbolTable.
Public routines (not to be confused with private routines) are tagged by the first letter of their selector: P for public routines.

Instance Variables:

	postBlock	<nil/BlockClosure>	the block specifying post actions of current method
									(nil if no post actions are specified by current method).

This instance variable acts as a container and a flag to ensure that post actions are completed before accepting next message.

Acknowledgements:
The initial implementation of POOL in a previous version of Actalk was designed by a group of students of DESS of University of Nantes, under the guidance of Jean Bezivin and Olivier Roux.'!


!PoolActivity methodsFor: 'accessing'!

postBlock: aBlock
	postBlock := aBlock! !

!PoolActivity methodsFor: 'events'!

kernelEventComplete: aMessage
	"Check if some post action has been specified.
	If the case, compute it just after performing the message (before final completion).
	Reset the post action block/flag for next message."
	"Note that this implementation does not provide a complete solution to the management of post actions.
	But this is sufficient thanks to the assumption that message computation is atomic.
	As a consequence no other message computation may rewrite current postBlock variable before completion of current message computation.
	See class ConcurrentSmalltalkActivity for a complete solution."

	postBlock isNil
		ifFalse:
			[postBlock value.
			postBlock := nil].
	super kernelEventComplete: aMessage! !

!PoolActivity methodsFor: 'message handling'!

acceptPublicRoutineMessage: aMessage
	"Accept and perform the public routine message as a standard Smalltalk message."

	^bself performMessage: aMessage! !

!PoolActivity methodsFor: 'default classes'!

addressClass
	"PoolActivity features (post actions and public routines)
	are useable teamed with classes PoolAddress and PoolObject.
	Meanwhile no incompatibility if not the case."

	^PoolAddress! !

ExplicitAcceptObject subclass: #PoolObject
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-Pool'!
PoolObject comment:
'Trio classes PoolObject, PoolActivity and PoolAddress implement POOL body model, explicit message acceptance, post actions, synchronous message passing, and (object-like) public routines.

See class PoolActivity comment.'!


!PoolObject methodsFor: 'body'!

body
	"Default Pool body is an endless loop to serially accept any message."

	| allSelectors |
	allSelectors := self class allScriptSelectors.
	[true] whileTrue:
		[self answer: allSelectors]! !

!PoolObject methodsFor: 'post actions'!

post: aBlock
	"Specify post actions (specified within a block) which are to be computed
	after completing the method and its return.
	This frees the sender from unneeded waiting, as post actions will take place after return.
	Typical use is for reinitialization."

	self activity postBlock: aBlock! !

!PoolObject methodsFor: 'default classes'!

activityClass
	"PoolObject user construct method
		post:
			calls
	PoolActivity method
		postBlock:"

	^PoolActivity! !

!PoolObject methodsFor: 'compatibility constraints'!

activityConstraint
	"Enforce constraint defined in method activityClass."

	^PoolActivity! !

