WithSenderObject subclass: #ActiveSemaphore
	instanceVariableNames: 'numberExcessSignals waitingProcessQueue '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-WithSender-Ex'!
ActiveSemaphore comment:
'Class ActiveSemaphore implements a semaphore as an active object.
This example is a revised version of the example described in ABCL/1 by Akinori Yonezawa et al. in the OOCP book, pages 70-71.
In this example the semaphore has implicit access to the sender of a wait message (a process active object) in order to possibly enqueue it into the queue of waiting processes.

Class ActiveSemaphore is defined as a subclass of active object behavior class WithSenderObject.

Instance variables:

	numberExcessSignals	<Integer>			number of signals in excess received.
	waitingProcessQueue	<OrderedCollection>	queueof processes waiting onto the semaphore.'!


!ActiveSemaphore methodsFor: 'initialize'!

initialize
	numberExcessSignals := 0.
	waitingProcessQueue := OrderedCollection new! !

!ActiveSemaphore methodsFor: 'script'!

signal
	"Signal the semaphore in order to signal/free first waiting process if any."

	waitingProcessQueue isEmpty
		ifTrue:
			["If no process is waiting, then increment the number of signals in excess."
			numberExcessSignals := numberExcessSignals + 1]
		ifFalse:
			["Otherwise, signal first waiting process."
			waitingProcessQueue removeFirst signal]!

wait
	"Sender wait onto the semaphore if no signal is in excess."
	"When the semaphore is signaled sender receives a signal message."

	numberExcessSignals > 0
		ifTrue:
			["If there are signals in excess, then current sender is immediately signaled."
			self sender signal.
			"And we decrement number of signals in excess."
			numberExcessSignals := numberExcessSignals - 1]
		ifFalse:
			["Otherwise, current sender is enqueued onto the waiting queue."
			waitingProcessQueue addLast: self sender]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ActiveSemaphore class
	instanceVariableNames: ''!


!ActiveSemaphore class methodsFor: 'instance creation'!

forMutualExclusion
	^self new signal!

new
	^super new initialize! !

!ActiveSemaphore class methodsFor: 'example'!

example: n
	"self example: 5"

	| semaphore |
	semaphore := self forMutualExclusion active.
	1 to: n do: [:i |
		(ActiveProcess new
				number: i semaphore: semaphore)
			active start]! !

WithSenderObject subclass: #ActiveProcess
	instanceVariableNames: 'number semaphore '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Ext-WithSender-Ex'!
ActiveProcess comment:
'Class ActiveProcess implements simple processes as active objects which can wait and signal an active object semaphore..
Computation of a process is extremely simplified into two methods/steps:
	method start, for starting computation and wait onto the semaphore,
	method signal, after resumption (signaling) by the semaphore.

ActiveProcess uses the generic events send (and receive) for tracing purposes.

Class ActiveProcess is defined as a subclass of active object behavior class WithSenderObject.

See class ActiveSemaphore comment.'!


!ActiveProcess methodsFor: 'initialize'!

number: n semaphore: sem
	number := n.
	semaphore := sem! !

!ActiveProcess methodsFor: 'script'!

signal
	self randomSleep.
	semaphore signal.
	self randomSleep!

start
	self randomSleep.
	semaphore wait! !

!ActiveProcess methodsFor: 'events'!

eventReceive: aMessage
	Transcript show: self printString , ' receive (' , aMessage compactPrintString , ')'; cr!

eventSend: aMessage to: anObject
	"This is just as a control of send generic event."

	Transcript show: self printString , ' send (' , aMessage compactPrintString , ') to ' , anObject printString; cr! !

!ActiveProcess methodsFor: 'private routines'!

randomSleep
	(Delay forSeconds: (Random new next * 5) rounded + 1) wait! !

!ActiveProcess methodsFor: 'printing'!

printOn: aStream
	"Print as its class name followed by its number."

	aStream nextPutAll: self class name.
	number printOn: aStream! !
