TrapMessagesObject subclass: #BecomeFuture
	instanceVariableNames: 'semaphore '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-FutureObjects'!
BecomeFuture comment:
'Class BecomeFuture implements future objects which become a value once computed.

Future objects are used to implement future type message passing, that is returning eagerly a promise of a value to be computed.
Class BecomeFuture is a subclass of class TrapMessagesObject (see below).

A future is implemented with a semaphore to control access to the future object (mainly wait until its assignment).

Note that a future object is implemented as a passive (standard) object and not an active one.
(See class FutureActor as an alternative).

Instance Variables:

	semaphore	<Semaphore>	synchronizes accesses to the value
								(accessing processes are suspended until the value is set).

Instance method reply: is the assignment method. We use the selector reply: because in future type message passing a future object becomes the reply destination of the requested computation.
Instance method value fetchs the value of the future and suspends until assignment.
Any other message sent to a future is implicitly (and transparently) forwarded to the value fetched (and thus suspended until assignment). This works because class Future is a subclass of class TrapMessagesObject and method doesNotUnderstand: is redefined as forwarding the message to the value.
Note that this forwarding/suspension disappears once the future has become the actual value.

Because the equality message (=) is defined in class TrapMessagesObject, we redefine it in class Future to forward = to the value fetched.

Note that there are two main (and actually strong) restrictions on this implementation of future objects through the primitive become:

	the value should not be an immediate object (e.g., an integer)
	as primitive become: cannot swap such primitive objects,

	the value should not be shared by other objects
	as it will become (be swapped with) the future object.

	(as an example, set the value of a BecomeFuture to be class Object, and see how Smalltalk-80 crashes...).

For the reasons explained above, this implementation may not be used freely.
See subclass Future as a general purpose class of future objects.'!


!BecomeFuture methodsFor: 'initialize'!

initialize
	semaphore := Semaphore new! !

!BecomeFuture methodsFor: 'accessing'!

reply: aValue
	"Become the value, and then free all pending value requests."
	"reply: is the method selector to set the value of the future object
	because a future object is intended to be used primarily
	as an implicit reply destination of a future type (eager reply) message send."
	"Warning: primitive become: will fail onto immediate objects (e.g., integers)."
	"Note: we create a temporary variable (sem) to refer to the semaphore,
	because after the become has been performed, the new identity of self (aValue)
	cannot access anymore to instance variables of the future."

	| sem |
	sem := semaphore.
	self become: aValue.
	[sem isEmpty] whileFalse:
		[sem signal]!

value
	"Consult the value of a future by waiting onto its semaphore.
	Then return self because the future has become the value
	before actual resumption of the method."

	semaphore wait.
	^self! !

!BecomeFuture methodsFor: 'implicit access'!

doesNotUnderstand: aMessage
	"Sending a message is transparently delegated to the value of the future.
	The future behaves as the value it will eventually hold."

	^self value performMessage: aMessage! !

!BecomeFuture methodsFor: 'comparing'!

= anObject
	"Transparently force access to the value."

	^self value = anObject! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

BecomeFuture class
	instanceVariableNames: ''!


!BecomeFuture class methodsFor: 'instance creation'!

new
	^super new initialize! !

BecomeFuture subclass: #Future
	instanceVariableNames: 'value '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-FutureObjects'!
Future comment:
'Class Future implements future objects which hold a value not computed yet.
Class Future is defined as a subclass of class BecomeFuture.
As opposed to BecomeFuture, class Future does not implement a become change.
It defines a variable, named value, to hold the actual value of the future.
It may hold any kind of object, including primitive ones, as opposed to class BecomeFuture.

Class Future is very close to class CBox of ConcurrentSmalltalk. (see in the book OOCP).

Instance Variables:

	value		<Object>		holds the value. May be not assigned yet.'!


!Future methodsFor: 'accessing'!

reply: aValue
	"Set the value of a future."
	"reply: is the method selector to set the value of the future object
	because a future object is intended to be used primarily
	as an implicit reply destination of a future type (eager reply) message send."

	value := aValue.
	semaphore signal!

value
	"Consult the value of a future by waiting onto its semaphore.
	Then reset the semaphore afterwards for next access."

	semaphore wait;
			signal.
	^value! !

Future subclass: #SAFuture
	instanceVariableNames: 'isReady isReadySemaphore '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-FutureObjects'!
SAFuture comment:
'Class SAFuture enforces single assignment of futures.
It is a subclass of class Future.

Instance Variables:

	isReady				<Boolean>		predicate telling if assignment has been already done.
	isReadySemaphore	<Semaphore>	protects assignment of the predicate above.'!


!SAFuture methodsFor: 'initialize'!

initialize
	super initialize.
	isReadySemaphore := Semaphore forMutualExclusion.
	isReadySemaphore critical:
		[isReady := false]! !

!SAFuture methodsFor: 'accessing'!

reply: aValue
	"Set the value of a future.
	A SAFuture enforces there is at most one assignment."

	isReadySemaphore critical:
		[isReady
			ifTrue:
				[self error: 'cannot be assigned twice']
			ifFalse:
				[isReady := true.
				super reply: aValue]]! !

!SAFuture methodsFor: 'printing'!

printOn: aStream
	"When value is set, print out as its value."

	isReady
		ifTrue: [value printOn: aStream]
		ifFalse: [super printOn: aStream]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SAFuture class
	instanceVariableNames: ''!


!SAFuture class methodsFor: 'counter example'!

example
	"SAFuture example"

	"Should raise an error."

	SAFuture new
		reply: 1;
		reply: 2! !

SharedQueue subclass: #MAFuture
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-FutureObjects'!
MAFuture comment:
'Class MAFuture (multiple assignment future) implements future objects which may hold several values (as in initial Abcl/1 model).
Class SharedQueue implements such basic functionalities, so class MAFuture is defined as a subclass of class SharedQueue.
We just need to rename the following accessing methods:

	nextValue	to:		next
	reply:		to:		nextPut:

and easily define the missing ones:

	isReady
	allValues'!


!MAFuture methodsFor: 'accessing'!

allValues
	"Fetch all values of the MAFuture.
	Remove all values of the shared queue."

	^[accessProtect critical:
		["Consume the number of elements = excessive signals."
		contents size timesRepeat: [readSynch wait].
		"Remove all elements into a collection."
		contents removeAllSuchThat: [:v | true]]] valueUninterruptably!

nextValue
	"Fetch next value element of the MAFuture."

	^self next!

reply: aValue
	"Set the value of a future. For a MAFuture this means enqueueing the value."

	self nextPut: aValue! !

!MAFuture methodsFor: 'predicates'!

isReady
	"Check if the MAFuture is ready, that is hold at least one value, that is not empty."

	^self isEmpty not! !

!MAFuture methodsFor: 'printing'!

printOn: aStream
	"Print as its class name, then its contents."

	super printOn: aStream.
	contents printOn: aStream! !
