Object subclass: #Basic1Activity
	instanceVariableNames: 'address bself '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Kernel-Basic'!
Basic1Activity comment:
'Class Basic1Activity is the basic class which implements the activity of an active object.
The activity is the object which specifies and controls selection and acceptance of messages and provides ressources (process(es)) to perform computation autonomously.

See more insights within class Activity comment.

Instance Variables:

	address	<Address>		the address of the active object.
	oself	<ActiveObject>	the behavior of the active object. (oself as object itself.)

Parameter methods, that is methods which may be redefined in extension subclasses in order to express various OOCP semantics, are:
	privateInitialize
	start
	createProcess
	body
	nextMessage
	acceptMessage:
	performMessage:
	addressClass
	invocationClassFor:'!


!Basic1Activity methodsFor: 'initialize'!

activeObject: anActiveObject
	bself := anActiveObject!

privateInitialize
	"This is a parameter method, intended to be possibly redefined in extension subclasses."
	"This initialization method is automatically called at the latest possible time,
	that is after initialization of instance variables has been automatically done
	and before activation starts."
	"Default initialize: do nothing."
	"It is not named initialize to ensure there is no name clash/shadowing with user (re)definition of initialize."! !

!Basic1Activity methodsFor: 'accessing'!

mailBox
	^address mailBox! !

!Basic1Activity methodsFor: 'address creation'!

computeAddressClass: activeObjectAddressClass
	"Compute and return the address class of the active object.
	Look for the most specialized address class
	(between the address class specified/computed by the active object behavior,
	and the one specified/computed by the activity itself)."

	^(activeObjectAddressClass
		sub: self addressClass
		ifUnrelated: [self error: 'inconsistency between address classes'])!

createAddressAndStart: activeObjectAddressClass
	"Create the address of the active object, initialize its mailbox, start the activity, and return the address."
	"Assume that method startActivity: does not specify a return value."

	^(self computeAddressClass: activeObjectAddressClass) new startActivity: self! !

!Basic1Activity methodsFor: 'activity setting'!

body
	"This is a parameter method, intended to be possibly redefined in extension subclasses."
	"Define the behavior body."
	"Default body is an endless loop to serially consume successive pending messages.
	That is default semantics of activity is reactivity with implicit acceptance of messages."

	[true] whileTrue:
		[self acceptNextMessage]!

createProcess
	"This is a parameter method, intended to be possibly redefined in extension subclasses."
	"Create and return the activity process which consumes the body."

	^[self body] newProcess!

start
	"This is a parameter method, intended to be possibly redefined in extension subclasses."
	"Start the activity by starting the process which consumes the body."

	self createProcess resume!

startWithAddress: anAddress
	"Initialize and start the activity.
	After initializing the address, (and before starting the process),
	perform the generic initialization."

	address := anAddress.				"The activity has reference to the address."
	bself initializeWithAddress: address.	"The active object behavior also does."
	self privateInitialize.					"Generic initialization."
	self start								"Start the (process implementing the) activity."! !

!Basic1Activity methodsFor: 'message handling'!

acceptMessage: aMessage
	"This is a parameter method, intended to be possibly redefined in extension subclasses."
	"Accept a message."
	"Default is to compute (perform) it."

	self performMessage: aMessage!

acceptNextMessage
	"Accepting next message."
	"(It gets suspended until there is one, see method nextMessage)."

	self acceptMessage: self nextMessage!

basicNextMessage
	"Remove next message from the mailbox."
	"Note that this fetching is suspended until the mailbox is not empty,
	thanks to the SharedQueue standard class."

	"This method is basic, that is it must not be redefined."

	^self mailBox next!

nextMessage
	"This is a parameter method, intended to be possibly redefined in extension subclasses."
	"Fetch next message to be computed."
	"Default is by removing next message from the mailbox."
	"Note that this fetching is suspended until the mailbox is not empty,
	thanks to the SharedQueue standard class."

	^self mailBox next!

performMessage: aMessage
	"This is a parameter method, intended to be possibly redefined in extension subclasses."
	"Perform (compute) a message."
	"Default is for the active object behavior to actually perform it and return the value."

	^bself performMessage: aMessage! !

!Basic1Activity methodsFor: 'default classes'!

addressClass
	"This is a parameter method, intended to be possibly redefined in extension subclasses."
	"Specify default address class."

	^Basic1Address!

invocationClassFor: aMessage
	"This is a parameter method, intended to be possibly redefined in extension subclasses."
	"It is used in order to make the invocation structure generic to possibly record further information,
	e.g, sender, reply destination, arrival time... managed by the system, e.g., class GenericSendAddress,
	as well as defined by the user, e.g., job length in class ShortestJobFirstPrinterActivity."
	"The parameter is the message received, because dispatch to an invocation class may depend on it.
	See examples of subclasses OptimizedImplicitReplyActivity and ShortestJobFirstPrinterActivity."

	"Default invocation class is standard class Message."

	^Message! !

Basic1Activity subclass: #Basic2Activity
	instanceVariableNames: 'process '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Kernel-Basic'!
Basic2Activity comment:
'Basic kernel class Basic2Activity offers a few more facilities than basic kernel class Basic1Activity.

Facilities are mainly:
	reference and control of the activity process,
	generic event methods (associated to invocation reception/acceptance/completion).

Instance Variables:

	process	<Process>	the process implementing the autonomy of the activity.'!


!Basic2Activity methodsFor: 'accessing'!

activeObject
	"Useful for the address to directly access to the active object behavior (through the activity)."

	^bself!

process
	^process! !

!Basic2Activity methodsFor: 'activity setting'!

createProcess
	"Provide a handle to the process."

	^process := super createProcess! !

!Basic2Activity methodsFor: 'process control'!

resume
	"Resume the activity process.
	Assume that the process has been suspended."

	process resume!

suspend
	"Suspend the activity process.
	Assume that the process is running."

	process suspend!

terminate
	"Terminate the activity process.
	Useful for cleanup (reclaiming resources)."

	process isNil
		ifFalse: [process terminate]! !

!Basic2Activity methodsFor: 'message handling'!

acceptMessage: aMessage
	"Add the triggering of generic event methods for accepting and completing computation of a message."

	self kernelEventAccept: aMessage.	
	super acceptMessage: aMessage.
	self kernelEventComplete: aMessage! !

!Basic2Activity methodsFor: 'events'!

kernelEventAccept: aMessage
	"This method is called when (before) the activity accepts a message."
	"This is the basic kernel method which should not be subclassed by application classes."

	bself eventAccept: aMessage!

kernelEventComplete: aMessage
	"This method is called once the activity completed computation of a message."
	"This is the basic kernel method which should not be subclassed by application classes."

	bself eventComplete: aMessage!

kernelEventReceive: aMessage
	"This method is called when the address has received a new message (and before actual enqueueing)."
	"This is the basic kernel method which should not be subclassed by application classes."

	bself eventReceive: aMessage! !

!Basic2Activity methodsFor: 'default classes'!

addressClass
	^Basic2Address! !

Object subclass: #Basic1ActiveObject
	instanceVariableNames: 'aself '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Kernel-Basic'!
Basic1ActiveObject comment:
'Class Basic1ActiveObject is the basic class which implements the behavior of an active object.
The behavior is the object which eventually consumes incoming messages and processes them.

See more insights within class ActiveObject comment.

Instance Variables:

	aself	<Address>	 the actual address of the active object. (aself as address of self.)

Parameter methods, that is methods which may be redefined in extension subclasses in order to express various OOCP semantics, are:
	privateInitialize
	active
	activityClass
	addressClass'!


!Basic1ActiveObject methodsFor: 'initialize'!

initializeWithAddress: anAddress
	"Initialize the address (aself) of the active object behavior.
	Then, perform the generic initialization."

	aself := anAddress.
	self privateInitialize!

privateInitialize
	"This is a parameter method, intended to be possibly redefined in extension subclasses."
	"This initialization method is automatically called at the latest possible time,
	that is after initialization of instance variables has been automatically done
	and before activation starts."
	"Default initialize: do nothing."
	"It is not named initialize to ensure there is no name clash/shadowing with user (re)definition of initialize."! !

!Basic1ActiveObject methodsFor: 'activity creation'!

active
	"This is a parameter method, intended to be possibly redefined in extension subclasses."
	"Create an active object by activating its behavior."
	"Default classes of the activity (and the address), e.g., classes Activity (and Address)
	are specified and returned by methods activityClass (and addressClass)."
	"This is similar to method defaultControllerClass of class View."

	^self activity: self activityClass address: self addressClass!

activity: activityClass address: addressClass
	"Create an active object whose classes of activity and address are specified."
	"Assume that method createAddressAndStart: returns the new address."

	^(activityClass new activeObject: self) createAddressAndStart: addressClass! !

!Basic1ActiveObject methodsFor: 'default classes'!

activityClass
	"This is a parameter method, intended to be possibly redefined in extension subclasses."
	"Specify default activity class."

	^Basic1Activity!

addressClass
	"This is a parameter method, intended to be possibly redefined in extension subclasses."
	"Specify default address class."

	^Basic1Address! !

Basic1ActiveObject subclass: #Basic2ActiveObject
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Kernel-Basic'!
Basic2ActiveObject comment:
'Basic kernel class Basic2ActiveObject offers a few more facilities than basic kernel class Basic1ActiveObject.

Facilities are mainly:
	check ensuring a single activation,
	control of the activity process,
	user-level generic event methods (associated to invocation reception/acceptance/completion),
	collecting all script selectors (see class method allScriptSelectors).

Note class Basic2ActiveObject also defines a fourth generic event method associated to message sending. It is only triggered when a message is sent to an active object class whose address is class SenderWithAddress or class InvocationAddress or one of their subclasses. Because the sender may be a standard Smalltalk-80 object, this generic method is also defined (empty) in class Object.'!


!Basic2ActiveObject methodsFor: 'accessing'!

activity
	"Useful shortcut when defining programming constructs which call on the activity."

	^aself activity!

address
	"Generic method returning the external reference to an object."
	"Useful for computing the address of a sender behavior active object."
	"Return the address of an active object."

	^aself! !

!Basic2ActiveObject methodsFor: 'activity creation'!

active
	"Check if the active object behavior has already been used (activated)."

	^aself isNil
		ifTrue: [super active]
		ifFalse: [self error: 'this active object has already been activated']! !

!Basic2ActiveObject methodsFor: 'process control'!

suspend
	"Suspend the activity process."

	aself suspend!

terminate
	"Terminate the activity process.
	Useful for cleanup (reclaiming resources)."

	aself terminate! !

!Basic2ActiveObject methodsFor: 'events'!

eventAccept: aMessage
	"This method is called when the active object accepts a message."
	"This method may be redefined in subclasses."
	"Default behavior is empty."

	"Below are possible examples of use to:

trace:
	Transcript show: self printString , ' accept (' , aMessage compactPrintString , ')'; cr.

schedule:
	Processor yield.

broadcast change to dependents (e.g., useful for MVC interface):
	self changed: #eventAccept with: aMessage.

step:
	self halt.
	"!

eventComplete: aMessage
	"This method is called once the active object completed computation of a message."
	"This method may be redefined in subclasses."
	"Default behavior is empty."

	"Below are possible examples of use to:

trace:
	Transcript show: self printString , ' complete (' , aMessage compactPrintString , ')'; cr.

schedule:
	Processor yield.

broadcast change to dependents (e.g., useful for MVC interface):
	self changed: #eventComplete with: aMessage.

step:
	self halt.
	"!

eventReceive: aMessage
	"This method is called once the active object has received a new message (and before actual enqueueing)."
	"This method may be redefined in subclasses."
	"Default behavior is empty."

	"Below are possible examples of use to:

trace:
	Transcript show: self printString , ' receive (' , aMessage compactPrintString , ')'; cr.

schedule:
	Processor yield.

broadcast change to dependents (e.g., useful for MVC interface):
	self changed: #eventReceive with: aMessage.

step:
	self halt.
	"!

eventSend: aMessage to: anAddress
	"This method is called once the active object has sent a message (and just before it actually gets received)
	to an active object whose address class is WithSenderAddress or InvocationAddress (or one of their subclasses).
	See method receiveMessage: of address class WithSenderAddress."
	"This method may be redefined in subclasses."
	"Default behavior is empty."

	"Below are possible examples of use to:

trace:
	Transcript show: self printString , ' send (' , aMessage compactPrintString , ') to ' , anAddress printString; cr.

schedule:
	Processor yield.

broadcast change to dependents (e.g., useful for MVC interface):
	self changed: #eventSend with: aMessage.

step:
	self halt.
	"! !

!Basic2ActiveObject methodsFor: 'default classes'!

activityClass
	^Basic2Activity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Basic2ActiveObject class
	instanceVariableNames: ''!


!Basic2ActiveObject class methodsFor: 'accessing method dictionary'!

allScriptSelectors
	"Answer the set of all the script selectors known by an active object behavior class."

	^self allSelectorsInCategory: #script untilClass: ActiveObject! !

TrapMessagesObject subclass: #Basic1Address
	instanceVariableNames: 'mailBox activity '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Kernel-Basic'!
Basic1Address comment:
'Class Basic1Address is the basic class which implements the address of an active object.
The address is the object which receives and buffers incoming messages.

Class Basic1Address is defined as a subclass of encapsulator class TrapMessagesObject as incoming messages will be trapped through the doesNotUnderstand: error.

See more insights within class Address comment.

Instance Variables:

	mailBox	<MailBox>	the mail queue containing incoming messages.
	activity	<Activity>	the activity of the active object.

Parameter methods, that is methods which may be redefined in extension subclasses in order to express various OOCP semantics, are:
	privateInitialize
	receiveMessage:
	asynchronousSend:inMessageQueue:'!


!Basic1Address methodsFor: 'initialize'!

initializeMailBox
	"Create a new mailbox, that is an extended shared queue, private to the address."
	"Is automatically called at creation time thanks to redefinition of new method."

	mailBox := MailBox new!

privateInitialize
	"This is a parameter method, intended to be possibly redefined in extension subclasses."
	"This initialization method is automatically called at the latest possible time,
	that is after initialization of instance variables has been automatically done
	and before activation starts."
	"Default initialize: do nothing."
	"It is not named initialize to ensure there is no name clash/shadowing with user (re)definition of initialize."! !

!Basic1Address methodsFor: 'accessing'!

mailBox
	^mailBox! !

!Basic1Address methodsFor: 'activity setting'!

startActivity: anActivity
	"Initialize reference to the activity, calls the generic initialization method, and then start the activity."

	activity := anActivity.
	self privateInitialize.
	activity startWithAddress: self! !

!Basic1Address methodsFor: 'error handling'!

doesNotUnderstand: aMessage
	"Receive the message."
	"Assumption of triggering the error is ensured by class TrapMessagesObject."

	^self receiveMessage: aMessage! !

!Basic1Address methodsFor: 'message passing'!

asynchronousSend: aMessage inMessageQueue: queue
	"This is a parameter method, intended to be possibly redefined in extension subclasses."
	"Generic asynchronous message sending."
	"Deafult behavior is to enqueue the message onto the specified message queue."

	queue nextPut: aMessage!

receiveMessage: aMessage
	"This is a parameter method, intended to be possibly redefined in extension subclasses."
	"Receive a message from an object."
	"Default interpretation of the reception is a standard asynchronous send."
	"Note that, asynchronousSend will return default return value, that is self,
	because an asynchronous and unidirectional send does not return a specific reply value."

	^self asynchronousSend: aMessage inMessageQueue: mailBox! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Basic1Address class
	instanceVariableNames: ''!


!Basic1Address class methodsFor: 'instance creation'!

new
	^super new initializeMailBox! !

Basic1Address subclass: #Basic2Address
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Kernel-Basic'!
Basic2Address comment:
'Basic kernel class Basic2Address offers a few more facilities than basic kernel class Basic1Address.

Facilities are mainly:
	(unsafe) direct access to the active object behavior,
	control of the activity process,
	printing external representation.'!


!Basic2Address methodsFor: 'accessing'!

activity
	^activity!

asObject
	"Return the behavior of an active object."
	"This bypasses the address external shell and thus it considers an active object as an almost standard Smalltalk object.
	This could be used to consult the contents of the active object behavior.
	This facility should be used extremely cautiously because it clashes with the encapsulation/serialization provided by the active object.
	Consequently it should be used to consult (read) but not to change (write) the state of the behavior.
	Examples of use are when building user interfaces onto active objects."

	^activity activeObject! !

!Basic2Address methodsFor: 'activity setting'!

startNewActivity: anActivity
	"As for method startActivity, but does not call the generic initialization method
	(which should be only performed only once when creating the initial activity)."
	"This method is useful to start new activities assigned to a single address, as for the Actor model of computation.
	See class ActorObject, in category Actalk-Ext-Actor."

	activity := anActivity.
	activity startWithAddress: self! !

!Basic2Address methodsFor: 'message passing'!

asynchronousSend: aMessage
	"Default/standard asynchronous message send form."

	self asynchronousSend: aMessage inMessageQueue: mailBox!

asynchronousSend: aMessage inMessageQueue: queue
	"Add the triggering of generic event when receiving a basic (asynchronous) message (before actual enqueueing)."

	"Note that the method kernelEventReceive: will also be defined in class UndefinedObject
	so that if there is no activity yet (for instance in case of FutureActors) nil does not signal an error."

	activity kernelEventReceive: aMessage.
	super asynchronousSend: aMessage inMessageQueue: queue!

internalReceiveMessage: aMessage
	"Resend the message by enqueueing it to the mailbox.
	Useful to implement (naive) delay of a message when the conditions of its acceptance are not yet fullfilled.
	(See for instance naive implementation of synchronization in class SelectiveAcceptActivity, category Actalk-Ext-SelectActivity).

	WARNING: This may result in an infinite accept/resend cycle. (See default protection below).

	Also notice that it does not call the method asynchronousSend:inMessageQueue:,
	this is because we don't want to trigger the receive event again."

	mailBox nextPut: aMessage.
	Processor yield		"Give way to other processes. This is NECESSARY.
						Otherwise there is strong chance this active object ALWAYS keep the processor.
						(By infinitely accepting and resending the same message)."!

internalReceiveMessageWithoutYield: aMessage
	"Resend the message by enqueueing it to the mailbox.
	Useful to implement (naive) delay of a message when the conditions of its acceptance are not yet fullfilled.

	WARNING: This MAY result in an infinite accept/resend cycle.
	No processor yield is performed. (NO PROTECTION!!).
	However this is useful when a yield is unneeded in case of no accept loop
	(this is the case for subprocesses computing only one message).
	(See for instance naive implementation of synchronization in method doesNotUnderstand:
	of class NullBoundedBuffer within category Actalk-Ext-Actor-Examples.)

	Also notice that it does not call the method asynchronousSend:,
	this is because we don't want to trigger the receive event again."

	mailBox nextPut: aMessage! !

!Basic2Address methodsFor: 'reset'!

reset
	"Full reset (mailbox and activity) and restart of an active object."

	activity terminate.
	self initializeMailBox;
		privateInitialize.
	activity startWithAddress: self! !

!Basic2Address methodsFor: 'process control'!

resume
	"Resume the activity process."

	activity resume!

suspend
	"Suspend the activity process."

	activity suspend!

terminate
	"Terminate  the activity process.
	Useful for cleanup (reclaiming resources)."

	activity terminate! !

!Basic2Address methodsFor: 'printing'!

printOn: aStream
	"External representation of an active object: & followed by the active object behavior. Ex: &a Counter."

	aStream nextPut: $&.
	activity activeObject printOn: aStream! !

