ActiveObject subclass: #JoinNOperator
	instanceVariableNames: 'n operator neutral values r '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Kernel-JoinCont'!
JoinNOperator comment:
'Class JoinNOperator represents a generic n-values operator join continuation.
It is an abstract class.
Join2Adder and Join2Multiplier are concrete derivations.

Instance Variables:

	n		<Integer>	the number of values required.
	operator	<Symbol>	the selector of the operation.
	neutral	<Integer>	the neutral element of the operation.
						It is used when applying the operation (assumed binary) to all values.
						Value is 0 (zero) for +, 1 (unity) for *.
	values	<OrderedCollection>	holds the values received.
	r		<Address>	the reply destination to which the computed result has to be sent.'!


!JoinNOperator methodsFor: 'initialize'!

initializeN: numberOfValues operator: anOperatorSymbol neutral: anInteger r: aReplyDestination
	n := numberOfValues.
	operator := anOperatorSymbol.
	neutral := anInteger.
	r := aReplyDestination.
	values := OrderedCollection new! !

!JoinNOperator methodsFor: 'script'!

reply: v
	values add: v.
	values size = n ifTrue:
		["If all values are received,
		then we may compute the final value and reply it."
		r reply:
			(values inject: neutral into: [:subTotal :next | subTotal perform: operator with: next]).
		"Optimize resource recovery, as no more message is expected."
		self terminate]! !

JoinNOperator subclass: #Join2Multiplier
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Kernel-JoinCont'!
Join2Multiplier comment:
'Class Join2Multiplier is a subclass of abstract class JoinNOperator.
It implements a binary multiplier join continuation.'!


!Join2Multiplier methodsFor: 'initialize'!

r: aReplyDestination
	self initializeN: 2 operator: #* neutral: 1 r: aReplyDestination! !

JoinNOperator subclass: #Join2Adder
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Kernel-JoinCont'!
Join2Adder comment:
'Class Join2Adder is a subclass of abstract class JoinNOperator.
It implements a binary adder join continuation.'!


!Join2Adder methodsFor: 'initialize'!

r: aReplyDestination
	self initializeN: 2 operator: #+ neutral: 0 r: aReplyDestination! !

ActiveObject subclass: #BinaryAdder
	instanceVariableNames: 'v1 r '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Kernel-JoinCont'!
BinaryAdder comment:
'Class BinaryAdder implements a (binary) additive join continuation.
A continuation represents the remaining part of a computation (as for this case summing up both sub computations) and can be specified as an explicit actor and passed along as a reply destination. A first simple example of continuation may be found in definition of class FactorialContinuation in category Actalk-Examples.
BinaryAdder join continuations are used for instance in class Fibonacci within category Actalk-Examples. Basically BinaryAdder will sum up the two recursive sub computations and send the result to the reply destination (either another embedding join continuation or the final destination, e.g., Print).

We assume that the operation (+) is commutative.
Note that Binary Adder also resynchronizes the two subcomputations and therefore acts as a join operator in data flow.

Instance Variables:

	v1		<nil/Integer>	will hold the first value received.
	r		<Address>	the reply destination to which the computed result has to be sent.
						It is stored in this variable once creating the continuation.'!


!BinaryAdder methodsFor: 'initialize'!

r: aReplyDestination
	r := aReplyDestination! !

!BinaryAdder methodsFor: 'script'!

reply: v
	"Receive one of the values."
	"We assume that a nil value means no value yet. Therefore we avoid using an extra flag.
	A cleaner specification with the behavior replacement model is found in class ActorBinaryAdder."

	v1 isNil	
		ifTrue: [v1 := v]			"First value is stored."
		ifFalse:
			[r reply: v1 + v.		"Second value triggers replying the computed result."
			self terminate]		"Optimize resource recovery, as no more message is expected."! !
