SharedQueue subclass: #MailBox
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Kernel-MailBox'!
MailBox comment:
'Class MailBox is a subclass of standard class SharedQueue.
SharedQueue provides basic functionalities of a mailbox to receive and retrieve messages (plus mutual exclusion and suspension while the mailbox is empty).
Class MailBox implements some further accessing and filtering methods specific to Actalk (looking for messages matching specific conditions and patterns, and incrementally looking for a message).'!


!MailBox methodsFor: 'accessing by index'!

atIndex: i
	"Look for and fetch the ith message in the mailbox.
	Suspend until there are at least i messages."
	"WARNING: this method is safe if only one activity process at a time makes such access.
	It is unsafe for concurrent use by many activity processes
	(because it makes assumption on the monotony of the mailbox)."
	"The manipulation of readSynch semaphore is somewhat tricky.
	We need to because we want to suspend until a new message comes
	BUT we don't want to decrement the number of items in the queue
	(until we actually find and remove a matching message)."

	| numberOfWaits |		"We need to record number of waits we may generate on readSynch semaphore"
	numberOfWaits := 0.		"in order to restore them afterwards."
	"This is because we need to keep the number of excessive signals which indicate the number of messages in the mailbox."
	[[accessProtect critical: [i > contents size]] valueUninterruptably]
		whileTrue:
			[readSynch wait.		"Wait for next incoming message."
			numberOfWaits := numberOfWaits + 1].
	numberOfWaits timesRepeat: [readSynch signal].	"Restore the semaphore excessive signals."
	^self basicAtIndex: i!

basicAtIndex: i
	"Look for and fetch the ith message in the mailbox.
	No check/suspension.
	We assume that there is at least i elements otherwise an error is raised."

	^[accessProtect critical: [contents at: i]] valueUninterruptably!

removeAtIndex: i
	"Remove and return the ith message of the mailbox."
	"It does assume that there is at least i messages in the mailbox."
	"WARNING: this method is safe if only one activity process at a time makes such access.
	It is unsafe for concurrent use by many activity processes
	(because it makes assumption on the monotony of the mailbox)."

	readSynch wait.		"We assume that we don't have to wait, but we want to tell we remove one item."
	^[accessProtect critical: [contents removeAtIndex: i]] valueUninterruptably! !

!MailBox methodsFor: 'accessing by condition'!

checkFirstMessageWithCondition: conditionBlock
	"Look for the first message satisfying the condition.
	As opposed to method firstMessageWithCondition: which is blocking,
	checkFirstMessageWithCondition: is not.
	If a matching message is found it removes and returns it. Otherwise it returns nil."

	^self internalFirstMessageWithCondition: conditionBlock withReadSynchWait: true!

firstMessageWithCondition: conditionBlock
	"Look for the first message satisfying the condition.
	(Condition is specified as a block with one parameter being the candidate message).
	It is blocking and suspend until it finds any matching one.
	(This means it suspends onto incoming messages as long as necessary)."
	"The manipulation of readSynch semaphore excess signals is somewhat tricky.
	We need to because we want to suspend until a new message comes
	BUT we don't want to decrement the number of items in the queue
	(until we actually find and remove a matching message)."

	| message numberOfWaits |
	"First instant look for a matching message."
	message := self internalFirstMessageWithCondition: conditionBlock withReadSynchWait: true.
	"We need to record number of waits on readSynch semaphore in order to restore them afterwards,
	because this number represents the number of messages in the mailbox."
	numberOfWaits := 0.
	"While no matching message has been found yet."
	[message isNil]
		whileTrue:
			["First wait for next incoming message."
			readSynch wait.
			numberOfWaits := numberOfWaits + 1.
			"The flag/argument of withReadSynchWait: is false, because we already waited for current message."
			message := self internalFirstMessageWithCondition: conditionBlock withReadSynchWait: false].
	"Restore the semaphore excessive signals (minus one for the removed item)."
	numberOfWaits - 1 timesRepeat: [readSynch signal].
	^message!

internalFirstMessageWithCondition: conditionBlock withReadSynchWait: isWait
	"Look for the first message satisfying the condition.
	It is not blocking and returns nil if no matching message is found."
	"The isWait flag tells if a wait onto the readSynch semaphore should be produced
	in case a matching message has been found and will be removed.
	Producing a wait is the usual case to tell that one item has been removed.
	However in some case (call from the loop of method firstMessageWithCondition:)
	the wait has already been produced, and doing it again would suspend computation."

	| message |
	[accessProtect critical:
		[message := contents detect: conditionBlock ifNone: [nil].
		message isNil				
			ifFalse:
				["If found, we remove (and then return) it."
				isWait ifTrue:
					["We know we don't have to wait, but we want to tell we remove one item."
					readSynch wait].	
				contents remove: message]]] valueUninterruptably.
	^message! !

!MailBox methodsFor: 'enumerating'!

detect: aBlock ifNone: exceptionBlock
	"Iterate aBlock over contents (ordered collection of messages) of the mailbox.
	Answer the first element for which aBlock evaluates to true.
	Atomic operation so that contents remains consistent during the iteration."

	^[accessProtect critical: [contents detect: aBlock ifNone: exceptionBlock]] valueUninterruptably!

do: aBlock
	"Iterate aBlock over contents (ordered collection of messages) of the mailbox.
	Atomic operation so that contents remains consistent during the iteration."

	^[accessProtect critical: [contents do: aBlock]] valueUninterruptably! !

!MailBox methodsFor: 'printing'!

printOn: aStream
	"Print as its class name, then its contents."

	super printOn: aStream.
	contents printOn: aStream! !
