Basic2Activity subclass: #Activity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Kernel'!
Activity comment:
'Kernel class Activity implements the activity of an active object.
The activity is the object which specifies and controls selection and acceptance of messages and provides ressources (process(es)) to perform computation autonomously.

To express and simulate a specific OOCP model, one will usually define a subclass of class Activity and redefine some of its parameter methods.

Parameter methods are defined in basic class Basic1Activity.

A specific class of active object behavior (user program) may also specify an associated activity class expressing its synchronization contraints (in order to achieve a better encapsulation). See examples in Actalk-Synchro-*-Ex* categories.

Useful facilities are following:

	cleanup facilities to provide a termination of activity process(es),
	see class methods cleanUp and allCleanUp,

	compatibility constraints to specify possible incompatibilities between the three components of an active object (behavior, activity and address),
	specification is provided by methods activeObjectConstraint and addressConstraint.'!


!Activity methodsFor: 'default classes'!

addressClass
	^Address! !

!Activity methodsFor: 'compatibility constraints'!

addressConstraint
	"Default bottom compatibility constraint."

	^Address!

objectConstraint
	"Default bottom compatibility constraint."

	^ActiveObject! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Activity class
	instanceVariableNames: ''!


!Activity class methodsFor: 'process cleanup'!

allCleanUp
	"Provide a reset/cleanup of activity processes of that kind.
	That is, terminate activity processes of all instances of this activity class and of its subclasses."

	"Following expression provides cleanup of all kinds of activities. Takes a little while.
	(On the Mac, move the mouse from time to time!!!!). Print should be reinitialized then."

	"Activity allCleanUp"

	Transcript show: 'CleanUp of ' , self name , ' hierarchy start'.
	self cleanUp.
	self allSubclasses do: [:anActivityClass |
		anActivityClass cleanUp.
		Transcript show: '.'].
	self postCleanUp.	"Some reinitialization may need to be done (to restart aborted activities)."
	Transcript show: ' done!!'; cr!

cleanUp
	"Provide a reset/cleanup of activity processes of this class."
	"See method allCleanUp to also cleanup all subclasses."

	"Note, since VisualWorks version 2.0,
	the GC does reclaim processes waiting on unreferenced semaphores.
	Thus this explicit cleanup facility is no more necessary. Thanks !!"

	self allInstances do:
		[:anActivity | anActivity terminate]!

gcTest: isCleanUp
	"ActiveObject gcTest: true"		"To show that processes are recovered by gc thanks to cleanup."
	"ActiveObject gcTest: false"	"To crash the system... and therefore prove the need for cleanup."

	"Test of cleanup to allow recovery of terminated processes.
	If cleanup is set (true), it should run infinitely. Stop it eventually by interrupt (^C).
	Otherwise (false), Smalltalk eventually crashes because processes cannot be recovered by gc."

	| n |
	n := 0.
	[true] whileTrue:
		[100 timesRepeat:
			[ActiveObject new active.
			n := n + 1].
		Transcript show: n printString , ' '.
		isCleanUp
			ifTrue: [ActiveObject cleanUp]]!

postCleanUp
	"We need to possibly reset the Print active object whose activity may have been terminated."

	(Activity isEqualOrSubclassOf: self)
		ifTrue: [ActiveTranscript initialize]! !

Basic2Address subclass: #Address
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Kernel'!
Address comment:
'Kernel class Address implements the address of an active object.
The address is the object which receives and buffers incoming messages.

To express and simulate a specific OOCP communication model, one will usually define a subclass of class Address and redefine some of its parameter methods to express different semantics of receiving messages.

The few parameter methods are defined in basic class Basic1Address.

See example of abstract subclass GenericSendAddress which parameterizes selection of various message passing types.

Useful facilities are following:

	compatibility constraints to specify possible incompatibilities between the three components of an active object (behavior, activity and address),
	specification is provided by methods activeObjectConstraint and activityConstraint.'!


!Address methodsFor: 'compatibility constraints'!

activityConstraint
	"Default bottom compatibility constraint."

	^Activity!

objectConstraint
	"Default bottom compatibility constraint."

	^ActiveObject! !

Basic2ActiveObject subclass: #ActiveObject
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Kernel'!
ActiveObject comment:
'Kernel class ActiveObject implements the behavior of an active object.
The behavior is the object which eventually consumes incoming messages and processes them.

User programs will define active objects as subclasses of this class (or one of its subclasses).

When defining new programming constructs (e.g., the waitFor: construct of ABCL/1 OOCP language), one will also define a new subclass of ActiveObject (e.g., class AbclObject).

Useful facilities are following:

	creating anonymous block continuation active objects as a more concise alternative to explicit classes of continuations,
	see methods continuationBlock: and singleReplyContinuationBlock:,

	user-event based tracing facilities which may automatically set and unset event based traces,
	see class methods setTrace: and setTraceOn*,

	cleanup facilities to provide a termination of activity process(es),
	see class methods cleanUp and allCleanUp,

	compatibility constraints to specify and check possible incompatibilities between the three components of an active object (behavior, activity and address),
	specification is provided by methods activityConstraint and addressConstraint,
	check up is provided by class methods checkConstraints and allCheckConstraints.

Please start with examples within the Actalk-Examples category.'!


!ActiveObject methodsFor: 'replying'!

reply: value withSelector: replySelector
	"Reply with a specific reply selector which needs to be computed.
	Basically it has been passed as an argument. See example in class Counter."

	self perform: replySelector with: value! !

!ActiveObject methodsFor: 'activity creation'!

active: activityClass
	"Create an active object with a specified activity class.
	This is useful when having various activity classes variants."

	^self activity: activityClass address: self addressClass! !

!ActiveObject methodsFor: 'continuation creation'!

continuationBlock: aBlock
	"Create a block continuation active object with the block (closure) as a behavior.
	The block has two arguments: the replied value, and the reference to the block continuation active object (usually named me)."
	"See class BlockContinuation."

	^(BlockContinuation new behaviorBlock: aBlock) active!

singleReplyContinuationBlock: aBlock
	"As for method continuationBlock:, except that the block continuation active object will accept a single reply message."
	"See class SingleReplyBlockContinuation."

	^(SingleReplyBlockContinuation new behaviorBlock: aBlock) active! !

!ActiveObject methodsFor: 'default classes'!

activityClass
	^Activity!

addressClass
	^Address! !

!ActiveObject methodsFor: 'compatibility constraints'!

activityConstraint
	"Default bottom compatibility constraint."

	^self activityClass!

addressConstraint
	"Default bottom compatibility constraint."

	^self addressClass! !

!ActiveObject methodsFor: 'version compatibility'!

actor
	"Compatibility with version 3.02 and before."

	Transcript show: self class name , ' should now use the selector active and not actor.'; cr.
	^self active! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ActiveObject class
	instanceVariableNames: ''!


!ActiveObject class methodsFor: 'event tracing'!

setTrace: setOrUnset
	"Set (or unset) tracing facilities for this class onto all four events.
	(aClass setTrace: true) to set, (aClass setTrace: false) to unset.
	It does not check whenever some event methods have already been defined.
	It simply adds them or removes them."

	self	setTraceOnReceive: setOrUnset;
		setTraceOnAccept: setOrUnset;
		setTraceOnComplete: setOrUnset;
		setTraceOnSend: setOrUnset!

setTraceOnAccept: setOrUnset
	"Set (or unset) tracing facility for this class onto the event accept.
	It does not check whenever an event methods has already been defined.
	It simply adds it or removes it."

	setOrUnset
		ifTrue: [self compile:
'eventAccept: aMessage
	Transcript show: self printString , '' accept ('' , aMessage compactPrintString , '')''; cr'
				classified: #'events']
		ifFalse: [self removeSelector: #eventAccept:]!

setTraceOnComplete: setOrUnset
	"Set (or unset) tracing facility for this class onto the event complete.
	It does not check whenever an event methods has already been defined.
	It simply adds it or removes it."

	setOrUnset
		ifTrue: [self compile:
'eventComplete: aMessage
	Transcript show: self printString , '' complete ('' , aMessage compactPrintString , '')''; cr'
				classified: #'events']
		ifFalse: [self removeSelector: #eventComplete:]!

setTraceOnReceive: setOrUnset
	"Set (or unset) tracing facility for this class onto the event receive.
	It does not check whenever an event methods has already been defined.
	It simply adds it or removes it."

	setOrUnset
		ifTrue: [self compile:
'eventReceive: aMessage
	Transcript show: self printString , '' receive ('' , aMessage compactPrintString , '')''; cr'
				classified: #'events']
		ifFalse: [self removeSelector: #eventReceive:]!

setTraceOnSend: setOrUnset
	"Set (or unset) tracing facility for this class onto the event send.
	It does not check whenever an event methods has already been defined.
	It simply adds it or removes it."

	setOrUnset
		ifTrue: [self compile:
'eventSend: aMessage to: anAddress
	Transcript show: self printString , '' send ('' , aMessage compactPrintString , '') to '' , anAddress printString; cr'
				classified: #'events']
		ifFalse: [self removeSelector: #eventSend:to:]!

unsetAllTraces
	"Remove ALL event tracing methods in the Actalk hierarchy.
	Useful for doing some cleanup like before filing out."

	"self unsetAllTraces"

	ActiveObject withAllSubclasses do:
		[:aClass | aClass setTrace: false]! !

!ActiveObject class methodsFor: 'process cleanup'!

allCleanUp
	"Provide a reset/cleanup of activity processes of that kind.
	Delegate allCleanUp to the associated activity class."

	"ActiveObject allCleanUp"

	| instance |						"To find the associated activity class,"
	instance := self someInstance.		"we need one of its instances,"
	instance isNil					"possibly by creating one."
		ifTrue: [instance := self basicNew].
	instance activityClass allCleanUp!

cleanUp
	"Provide a reset/cleanup of activity processes associated to active objects of this class.
	Delegate cleanUp to the associated activity class."
	"See method allCleanUp to also cleanup all subclasses."

	| instance |						"To find the associated activity class,"
	instance := self someInstance.		"we need one of its instances,"
	instance isNil					"possibly by creating one."
		ifTrue: [instance := self new].
	instance activityClass cleanUp! !

!ActiveObject class methodsFor: 'compatibility constraints'!

allCheckConstraints
	"Check constraints for all active object classes."

	"ActiveObject allCheckConstraints"

	| isSatisfied |
	isSatisfied := true.
	Transcript show: 'Constraints check for ' , self name , ' hierarchy start'.
	self checkConstraints.
	self allSubclasses do: [:anActiveObjectClass |
		isSatisfied := isSatisfied & (anActiveObjectClass checkConstraints).
		Transcript show: '.'].
	Transcript show: ' done!!'; cr.
	isSatisfied
		ifTrue: [Transcript show: 'All classes of ' , self name , ' hierarchy are compatible.'; cr]
		ifFalse: [Transcript show: 'Some classes of ' , self name , ' hierarchy are incompatible.';
						cr;
 						show: 'Please check the Transcript window.'].
	^isSatisfied!

checkConstraint: class1 with: class2
	"Check one compatibility constraint by testing if class1 is equal or a subclass of class2.
	(class2 may be an ordered collection of classes.)
	Return the boolean result of the check."

	| isSatisfied |
	isSatisfied :=
		class2 isBehavior
			ifTrue:
				["Only one class as a constraint."
				class1 isEqualOrSubclassOf: class2]
			ifFalse:
				[class2 isSequenceable
					ifFalse:
						["The constraint is neither a class nor an ordered collection of classes."
						self error: 'compatibility constraint not valid']
					ifTrue:
						["The constraint is an ordered collection of classes.
						At least one of them should be verified."
						class2 inject: false into: [:boolean :class |
												boolean | (class1 isEqualOrSubclassOf: class)]]].
	isSatisfied ifFalse:
		[Transcript show: 'constraint compatibility violation : '
						, class1 name , ' should be a subclass of ' , class2 printString; cr].
	^isSatisfied!

checkConstraints
	"Check if active object class, activity class, and address class are mutually compatible."

	| isSatisfied object activity address |
	"We need to create an instance, and its associated activity and address."
	"Note that we are using basicNew in order to bypass any initialization method."
	object := self basicNew.					
	activity := object activityClass basicNew.
	address := (activity computeAddressClass: object addressClass) basicNew.
	isSatisfied :=		"Conjonction of all 6 constraint checks."
	"Check constraints specified by the active object class."
		(self checkConstraint: activity class with: object activityConstraint)
		& (self checkConstraint: address class with: object addressConstraint)
	"Check constraints specified by the activity class."
		& (self checkConstraint: object class with: activity objectConstraint)
		& (self checkConstraint: address class with: activity addressConstraint)
	"Check constraints specified by the address class."
		& (self checkConstraint: object class with: address objectConstraint)
		& (self checkConstraint: activity class with: address activityConstraint).
	isSatisfied ifFalse:
		[Transcript show: self name
				, ' has compatibility problem with: '
				, activity class name , ' and ' , address class name; cr].
	^isSatisfied! !

