SynchroConcurrentActivity subclass: #CountersActivity
	instanceVariableNames: 'receivedDictionary acceptedDictionary completedDictionary '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Counters'!
CountersActivity comment:
'Activity class CountersActivity implements synchronization counters.

Synchronization counters record status of invocations of methods. Usually three events are recorded for each method and are counted by corresponding counters: reception, acceptance, and completion.
These are exactly the three synchronization generic events. Consequently mapping of incrementation of corresponding counters is immediate.
Therefore class CountersActivity is defined as a subclass of activity class SynchroConcurrentActivity.

Synchronization constraints are expressed through guards as for class GuardsActivity.
Meanwhile the guards may refer to synchronization counters in order to express constraints based on the status of other invocations. This provides precise control of concurrency of invocations, e.g., possible mutual exclusion between some methods.
See examples in category Actalk-Synchro-Counters-Ex.

Note that the implementation of class CountersActivity relies on resending not acceptable messages. As a consequence it may loop for ever possibly rechecking guards of looping messages although conditions cannot change (and therefore don''t need to be reevaluated) if no event (new message) occurs.
Subcass InvocationActivity provides a sophisticated implementation with a more precise reevaluation semantics for guards.

The initial implementation of synchronization counters in a previous version of Actalk was designed by a group of students of DESS of University of Nantes, under the guidance of Jean Bezivin and Olivier Roux.'!


!CountersActivity methodsFor: 'initialize'!

makeSynchroCounterDictionariesOnSelectors: selectors
	"Create the three synchronization counters dictionaries with an entry for each selector."

	receivedDictionary := IdentityDictionary new.
	acceptedDictionary := IdentityDictionary new.
	completedDictionary := IdentityDictionary new.
	selectors do: [:selector |
		receivedDictionary at: selector put: 0.
		acceptedDictionary at: selector put: 0.
		completedDictionary at: selector put: 0]!

privateInitialize
	"Create the three synchronization counters dictionaries."

	super privateInitialize.
	self makeSynchroCounterDictionariesOnSelectors: bself class allScriptSelectors! !

!CountersActivity methodsFor: 'message handling'!

isCandidateMessage: aMessage
	"A message is candidate for acceptance if its asociated guard evaluates to true."
	"Note that checking the acceptance AND the accept synchronization event (if the actual check is true) must be atomic."

	^synchroMutualExclusion critical:
		[(self evaluateGuardForMessage: aMessage)
			ifTrue:
				[self synchroEventAccept: aMessage.
				true]
			ifFalse:
				[false]]!

nextMessage
	"Next message is first acceptable message.
	Resend other ones."

	| message |
	[message := super nextMessage.
	self isCandidateMessage: message] whileFalse:
		[address internalReceiveMessage: message].
	^message! !

!CountersActivity methodsFor: 'guard testing'!

evaluateGuardForMessage: aMessage
	"Evaluate the associated condition/guard with the current arguments."

	^self perform: (self guardOfSelector: aMessage selector)
		withArguments: aMessage arguments!

guardOfSelector: selector
	"Return the selector (symbol) of the guard associated to a selector."
	"A guard is simply associated to a selector with the prefix guardOF.
	For instance the guard of method put: is the method guardOFput:."

	^('guardOF' , selector) asSymbol! !

!CountersActivity methodsFor: 'synchro events'!

synchroEventAccept: aMessage
	"Increment the number of accepted invocations of current selector."

	super synchroEventAccept: aMessage.
	self incrAccepted: aMessage selector!

synchroEventComplete: aMessage
	"Increment the number of completed invocations of current selector."

	super synchroEventComplete: aMessage.
	self incrCompleted: aMessage selector!

synchroEventReceive: aMessage
	"Increment the number of received invocations of current selector."

	super synchroEventReceive: aMessage.
	self incrReceived: aMessage selector! !

!CountersActivity methodsFor: 'counters accessing'!

accepted: selector
	"Number of accepted invocations of message selector."

	^acceptedDictionary at: selector!

completed: selector
	"Number of completed invocations of message selector."

	^completedDictionary at: selector!

current: selector
	"Number of current (accepted but not completed yet, that is currently being computed) invocations of message selector."

	^(self accepted: selector) - (self completed: selector)!

pending: selector
	"Number of pending (received but not accepted yet) invocations of message selector."

	^(self received: selector) - (self accepted: selector)!

received: selector
	"Number of received invocations of message selector."

	^receivedDictionary at: selector! !

!CountersActivity methodsFor: 'counters updating'!

incrAccepted: selector
	"Increment the number of accepted invocations of message selector."

	acceptedDictionary
		at: selector
		put: (acceptedDictionary at: selector) + 1!

incrCompleted: selector
	"Increment the number of completed invocations of message selector."

	completedDictionary
		at: selector
		put: (completedDictionary at: selector) + 1!

incrReceived: selector
	"Increment the number of received invocations of message selector."

	receivedDictionary
		at: selector
		put: (receivedDictionary at: selector) + 1! !

ImplicitReplyObject subclass: #CountersObject
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Counters'!
CountersObject comment:
'Class CountersObject is a class of active object behaviors. Its only purpose is to dispatch onto activity class CountersActivity.
It is defined as a subclass of behavior class ImplicitReplyObject in order to achieve consistency with its associated activity class hierarchy. (Meanwhile it could have been as well defined as a subclass of ActiveObject).'!


!CountersObject methodsFor: 'default classes'!

activityClass
	"Only purpose is to dispatch onto activity class CountersActivity."

	^CountersActivity! !

