GuardsObject subclass: #GuardsTable
	instanceVariableNames: 'n p numberSteps philosophers forksAvailability '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Guards-Ex'!
GuardsTable comment:
'Class GuardsTable implements the shared ressource of philosophers, that is forks placed onto the table. See class Philosopher comment.'!


!GuardsTable methodsFor: 'initialize'!

initializeNumberForks: numberForks numberPhilosophersPerFork: numberPhilosophersPerFork numberSteps: steps
	n := numberForks.
	p := numberPhilosophersPerFork.
	numberSteps := steps.
	self makeForksAvailability!

makeForksAvailability
	"Create and initialize the array containing the availability of each fork."

	forksAvailability := Array new: n withAll: true!

makePhilosophers
	"Create and initialize the n * p philosophers.
	There are p philosophers for each position around the table."

	philosophers := Array new: n*p.
	1 to: n*p do:
		[:i | philosophers at: i put:
			(Philosopher new
				number: i
				position: i - 1 \\ n + 1
				n: n
				table: aself
				numberSteps: numberSteps)
					active]! !

!GuardsTable methodsFor: 'script'!

pickUp: i and: j by: philosopher
	forksAvailability
		at: i put: false;
		at: j put: false!

putDown: i and: j by: philosopher
	forksAvailability
		at: i put: true;
		at: j put: true!

start
	self makePhilosophers! !

!GuardsTable methodsFor: 'predicates'!

areForksAvailableAt: i and: j
	^(forksAvailability at: i) & (forksAvailability at: j)! !

!GuardsTable methodsFor: 'default classes'!

activityClass
	"Synchronization is specified by the activity guards specification."

	^GuardsTableActivity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GuardsTable class
	instanceVariableNames: ''!


!GuardsTable class methodsFor: 'example'!

example: n
	"self example: 5"

	(self new initializeNumberForks: 5 numberPhilosophersPerFork: 2 numberSteps: 3) active start! !

IRBoundedBuffer subclass: #Guards1BoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Guards-Ex'!
Guards1BoundedBuffer comment:
'Class Guards1BoundedBuffer implements the example of the bounded buffer with guards to specify synchronization.
Guard specification is expressed within the associated activity class in order to enforce a clear separation between program and control.
Note that because guard specifications are quite modular, they behave quite well against inheritance anomaly.'!


!Guards1BoundedBuffer methodsFor: 'default classes'!

activityClass
	"Synchronization is specified by the activity guards specification."

	^Guards1BoundedBufferActivity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Guards1BoundedBuffer class
	instanceVariableNames: ''!


!Guards1BoundedBuffer class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	super exampleSize: maxSize numberItems: numberItems speedRatio: ratio! !

ActiveObject subclass: #Philosopher
	instanceVariableNames: 'number leftForkPosition rightForkPosition table numberSteps '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Guards-Ex'!
Philosopher comment:
'Class Philosopher implements the dining philosophers example with guards.
As opposed to the solution with POOL explicit message acceptance (see class PoolPhilosopher) which needed to restrict the number of philosophers accessing the table in order to ensure deadlock avoidance, this description does not need to.
We even may run any number of philosophers, e.g., p times the number of forks.
The solution makes use of a class GuardsTable to centralize access to the forks in order to ensure that a philosopher picks up his two forks at once (atomically) if they are both available.'!


!Philosopher methodsFor: 'initialize'!

number: num position: i n: n table: aTable numberSteps: steps
	number := num.
	leftForkPosition := i.
	rightForkPosition := (leftForkPosition \\ n) + 1.
	table := aTable.
	numberSteps := steps! !

!Philosopher methodsFor: 'body'!

body
	"Body: a philosopher does numberSteps times his life steps, that is: think and eat."
	"Note that this active object class has no script method as it receives no message."

	1 to: numberSteps do: [: i |
		self think.
		Transcript show: self printString , ' wants to eat for the ' , i printString , 'th time and requests forks '
			, leftForkPosition printString , ' and ' , rightForkPosition printString; cr.
		"He requests his forks. Note that he waits (value) for the acknowledgement."
		(table pickUp: leftForkPosition and: rightForkPosition by: aself) value.
		self eat.
		"Note that when releasing forks, he does not need to wait for acknowledgement to proceed."
		table putDown: leftForkPosition and: rightForkPosition by: aself.
		Transcript show: self printString , ' has finished eating and has released forks '
			, leftForkPosition printString , ' and ' , rightForkPosition printString; cr]! !

!Philosopher methodsFor: 'private routines'!

eat
	| delay |
	delay := (Random new next * 5) rounded + 1.
	Transcript show: self printString , ' eats with forks '
			, leftForkPosition printString , ' and ' , rightForkPosition printString
			, ' during ' , delay printString , ' seconds.'; cr.
	(Delay forSeconds: delay) wait!

think
	| delay |
	delay := (Random new next * 5) rounded + 1.
	Transcript show: self printString , ' thinks during ' , delay printString , ' seconds.'; cr.
	(Delay forSeconds: delay) wait! !

!Philosopher methodsFor: 'printing'!

printOn: aStream
	"Print as its class name followed by its number."

	aStream nextPutAll: self class name.
	number printOn: aStream! !

!Philosopher methodsFor: 'default classes'!

activityClass
	"Activity of a philosopher is described by its body."

	^ObjectBodyActivity! !

Get2BoundedBuffer subclass: #GuardsGet2BoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Guards-Ex'!
GuardsGet2BoundedBuffer comment:
'This class implements an example of inheritance anomaly.'!


!GuardsGet2BoundedBuffer methodsFor: 'state predicates'!

hasAtLeastTwoElements
	^contents size >= 2! !

!GuardsGet2BoundedBuffer methodsFor: 'default classes'!

activityClass
	"Synchronization is specified by the activity guards specification."

	^GuardsGet2BoundedBufferActivity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GuardsGet2BoundedBuffer class
	instanceVariableNames: ''!


!GuardsGet2BoundedBuffer class methodsFor: 'example'!

exampleGet2Size: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleGet2Size: 3 numberItems: 8 speedRatio: 3"
	"self exampleGet2Size: 3 numberItems: 8 speedRatio: 1/3"

	super exampleGet2Size: maxSize numberItems: numberItems speedRatio: ratio! !

GuardsActivity subclass: #GuardsBoundedBufferActivity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Guards-Ex'!
GuardsBoundedBufferActivity comment:
'This activity class specifies the synchronization of its associated behavior class.'!


!GuardsBoundedBufferActivity methodsFor: 'guards'!

guardOFget
	"Not empty."

	^bself isEmpty not!

guardOFput: item
	"Not full."

	^bself isFull not! !

GuardsBoundedBufferActivity subclass: #GuardsGet2BoundedBufferActivity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Guards-Ex'!
GuardsGet2BoundedBufferActivity comment:
'This activity class specifies the synchronization of its associated behavior class.'!


!GuardsGet2BoundedBufferActivity methodsFor: 'guards'!

guardOFget2
	"At least two elements."

	^bself hasAtLeastTwoElements! !

GuardsBoundedBufferActivity subclass: #GuardsGgetBoundedBufferActivity
	instanceVariableNames: 'afterPut '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Guards-Ex'!
GuardsGgetBoundedBufferActivity comment:
'This activity class specifies the synchronization of its associated behavior class.'!


!GuardsGgetBoundedBufferActivity methodsFor: 'initialize'!

privateInitialize
	super privateInitialize.
	afterPut := false! !

!GuardsGgetBoundedBufferActivity methodsFor: 'events'!

kernelEventComplete: aMessage
	super kernelEventComplete: aMessage.
	afterPut := aMessage selector = #put:! !

!GuardsGgetBoundedBufferActivity methodsFor: 'guards'!

guardOFgget
	"Not empty and not after a put."

	^(self guardOFget) & (afterPut not)! !

Guards1BoundedBuffer subclass: #GuardsBoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Guards-Ex'!
GuardsBoundedBuffer comment:
'Class GuardsBoundedBuffer is analog to class Guards1BoundedBuffer but with the optimized implementation of message selection of activity class GuardsActivity.'!


!GuardsBoundedBuffer methodsFor: 'default classes'!

activityClass
	"Synchronization is specified by the activity guards specification."

	^GuardsBoundedBufferActivity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GuardsBoundedBuffer class
	instanceVariableNames: ''!


!GuardsBoundedBuffer class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	super exampleSize: maxSize numberItems: numberItems speedRatio: ratio! !

GuardsActivity subclass: #GuardsTableActivity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Guards-Ex'!
GuardsTableActivity comment:
'Class of activity TableActivity defines synchronization of accesses to forks of a GuardsTable.'!


!GuardsTableActivity methodsFor: 'guards'!

guardOFpickUp: i and: j by: philosopher
	"Both forks must be available."

	^bself areForksAvailableAt: i and: j!

guardOFputDown: i and: j by: philosopher
	"Always free to release forks."

	^true!

guardOFstart
	"Free start."

	^true! !

GgetBoundedBuffer subclass: #GuardsGgetBoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Guards-Ex'!
GuardsGgetBoundedBuffer comment:
'This class implements an example of inheritance anomaly.'!


!GuardsGgetBoundedBuffer methodsFor: 'default classes'!

activityClass
	"Synchronization is specified by the activity guards specification."

	^GuardsGgetBoundedBufferActivity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GuardsGgetBoundedBuffer class
	instanceVariableNames: ''!


!GuardsGgetBoundedBuffer class methodsFor: 'example'!

exampleGgetSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleGgetSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleGgetSize: 3 numberItems: 8 speedRatio: 1/3"

	super exampleGgetSize: maxSize numberItems: numberItems speedRatio: ratio! !

Guards1Activity subclass: #Guards1BoundedBufferActivity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Guards-Ex'!
Guards1BoundedBufferActivity comment:
'This activity class specifies the synchronization of its associated behavior class.'!


!Guards1BoundedBufferActivity methodsFor: 'guards'!

guardOFget
	"Not empty."

	^bself isEmpty not!

guardOFput: item
	"Not full."

	^bself isFull not! !

