ConcurrentBoundedBuffer subclass: #FCFSBoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
FCFSBoundedBuffer comment:
'Class FCFSBoundedBuffer implements the example of the bounded buffer with assumption of message preservation ordering (FCFS stands for first come first served).
See its associated activity class FCFSBoundedBufferActivity.'!


!FCFSBoundedBuffer methodsFor: 'accessing'!

maxSize
	^maxSize! !

!FCFSBoundedBuffer methodsFor: 'default classes'!

activityClass
	"Synchronization is specified by the activity guards specification."

	^FCFSBoundedBufferActivity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

FCFSBoundedBuffer class
	instanceVariableNames: ''!


!FCFSBoundedBuffer class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	super exampleSize: maxSize numberItems: numberItems speedRatio: ratio! !

InvocationActivity subclass: #FCFSBoundedBufferActivity
	instanceVariableNames: 'maxSize '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
FCFSBoundedBufferActivity comment:
'This activity class specifies the synchronization of its associated behavior class.'!


!FCFSBoundedBufferActivity methodsFor: 'initialize'!

privateInitialize
	super privateInitialize.
	maxSize := bself maxSize! !

!FCFSBoundedBufferActivity methodsFor: 'accessing'!

numberOfElements
	"Number of elements currently in the buffer.
	Compute the difference between completed put: and completed get.
	Note: this is correct because we assume that there is only one put: at once, and only one get at once."

	^(self completed: #put:) - (self completed: #get)! !

!FCFSBoundedBufferActivity methodsFor: 'guards'!

guardOFget
	"Only one get at once and the buffer should be not empty.
	Also ensures message ordering preservation (no prior pending get)."

	^(self noCurrent: #get)
		and: [self numberOfElements > 0
			and: [self noPending: #get priorTo: currentInvocation arrivalTime]]!

guardOFput: item
	"Only one put at once and the buffer should not be full.
	Also ensures message ordering preservation (no prior pending put:)."

	^(self noCurrent: #put:)
		and: [self numberOfElements < maxSize
			and: [self noPending: #put: priorTo: currentInvocation arrivalTime]]! !

InvocationActivity subclass: #PriorityPrinterActivity
	instanceVariableNames: ''
	classVariableNames: 'PriorityTable '
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
PriorityPrinterActivity comment:
'Activity class PriorityPrinterActivity implements the synchronization of the dynamic priority printer example.
Method updatePriorityOf:to: has a true guard. Actual reassignment of the priority takes place within the acceptance event. Consequently this assignment is implicitly in mutual exclusion with reading of the priority table from within a guard by a printing invocation.'!


!PriorityPrinterActivity methodsFor: 'initialize'!

initialize
	PriorityTable := Dictionary new! !

!PriorityPrinterActivity methodsFor: 'guards'!

guardOFprint: job length: length group: group
	"One printing job at a time and no pending print invocation with a higher priority
	or with the same priority but with some prior arrival time."

	^(self noCurrent: #print:length:group:)
		and: [self noPending: #print:length:group: with: [:invocation |
				(PriorityTable at: (invocation arg: 3)) > (PriorityTable at: group)
					or: [(PriorityTable at: (invocation arg: 3)) = (PriorityTable at: group)
						and: [invocation arrivalTime < currentInvocation arrivalTime]]]]!

guardOFupdatePriorityOf: group to: newPriority
	"No restriction as update of the table is managed by the accept event.
	As a consequence consistency of the priority table is ensured by atomicity (mutual exclusion) of events and guard testing."

	^true! !

!PriorityPrinterActivity methodsFor: 'synchro events'!

synchroEventAccept: anInvocation
	"If this is a priority updating request, perform the reassignment."

	super synchroEventAccept: anInvocation.
	anInvocation ifSelector: #updatePriorityOf:to: do:
		[PriorityTable at: (anInvocation arg: 1) put: (anInvocation arg: 2)]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

PriorityPrinterActivity class
	instanceVariableNames: ''!


!PriorityPrinterActivity class methodsFor: 'instance creation'!

new
	^super new initialize! !

ImplicitReplyObject subclass: #AlarmClock
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
AlarmClock comment:
'Class AlarmClock describes the alarm clock problem.

The alarm clock should wake up after a certain amount of time (represented as a number of ticks received).
Note that methods of AlarmClock have empty bodies, because this is a pure synchronization problem. See associated activity class AlarmClockActivity.

This solution has been initially described in the report on Synchronization Variables (see class InvocationActivity comment).'!


!AlarmClock methodsFor: 'script'!

tick
	"No code, only synchronization."

	Transcript show: self printString , ' tick!!'; cr!

wakeUpAfter: time
	"No code, only synchronization."

	Transcript show: self printString , ' wakes up!!'; cr! !

!AlarmClock methodsFor: 'default classes'!

activityClass
	"Synchronization is specified by the activity guards specification."

	^AlarmClockActivity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

AlarmClock class
	instanceVariableNames: ''!


!AlarmClock class methodsFor: 'example'!

example: numberOfTicks wakeUpFrom: t1 after: delay
	"self example: 10 wakeUpFrom: 3 after: 5"

	| alarmClock |
	alarmClock := self new active.
	[numberOfTicks timesRepeat:
		[alarmClock tick.
		(Delay forSeconds: 1) wait]] fork.
	(Delay forSeconds: t1) wait.
	alarmClock wakeUpAfter: delay! !

ImplicitReplyObject subclass: #DiskHeadScheduler
	instanceVariableNames: ''
	classVariableNames: 'NumberOfTracks '
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!


!DiskHeadScheduler methodsFor: 'accessing'!

numberOfTracks
	^NumberOfTracks! !

!DiskHeadScheduler methodsFor: 'script'!

moveHeadTo: destinationTrackNumber andTransfer: data
	"Move the disk head to destinationTrackNumber and transfer data to the disk."! !

!DiskHeadScheduler methodsFor: 'events'!

eventAccept: aMessage
	Transcript show: self printString , ' accept (' , aMessage compactPrintString , ')'; cr! !

!DiskHeadScheduler methodsFor: 'default classes'!

activityClass
	"Synchronization is specified by the activity guards specification."

	^DiskHeadSchedulerActivity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

DiskHeadScheduler class
	instanceVariableNames: ''!


!DiskHeadScheduler class methodsFor: 'initialize'!

initialize
	NumberOfTracks := 100! !

!DiskHeadScheduler class methodsFor: 'example'!

elevatorExample: numberOfMoves
	"self elevatorExample: 20"

	| random trackNumber disk |
	random := Random new.
	disk := self new active: ElevatorDiskHeadSchedulerActivity.
	1 to: numberOfMoves do: [:i |
		trackNumber := (random next * NumberOfTracks) truncated.
		disk moveHeadTo: trackNumber andTransfer: i]!

example: numberOfMoves
	"self example: 20"

	| random trackNumber disk |
	random := Random new.
	disk := self new active.
	1 to: numberOfMoves do: [:i |
		trackNumber := (random next * NumberOfTracks) truncated.
		disk moveHeadTo: trackNumber andTransfer: i]!

nearestJobNextExample: numberOfMoves
	"self nearestJobNextExample: 20"

	| random trackNumber disk |
	random := Random new.
	disk := self new active: NearestJobNextDiskHeadSchedulerActivity.
	1 to: numberOfMoves do: [:i |
		trackNumber := (random next * NumberOfTracks) truncated.
		disk moveHeadTo: trackNumber andTransfer: i]!

optimizedElevatorExample: numberOfMoves
	"self optimizedElevatorExample: 20"

	| random trackNumber disk |
	random := Random new.
	disk := self new active: OptimizedElevatorDiskHeadSchedulerActivity.
	1 to: numberOfMoves do: [:i |
		trackNumber := (random next * NumberOfTracks) truncated.
		disk moveHeadTo: trackNumber andTransfer: i]! !

PlainInvocation subclass: #SkippedInvocation
	instanceVariableNames: 'numberTimesSkipped '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
SkippedInvocation comment:
'Invocation class SkippedInvocation adds an instance variable (numberTimesSkipped) to record the number of times the invocation has been skipped.
This is useful to ensure a starvation free policy.
Class SkippedInvocation is defined as a subclass of invocation class PlainInvocation.
See activity class StarvationFreeTableActivity.'!


!SkippedInvocation methodsFor: 'initialize'!

initialize
	numberTimesSkipped := 0! !

!SkippedInvocation methodsFor: 'accessing'!

numberTimesSkipped
	^numberTimesSkipped!

skip
	numberTimesSkipped := numberTimesSkipped + 1

	"Warning: the tracing code below deadlocks the demo if evaluated.
	This is very bizarre indeed...
	The request on the transcript waits indefinitely for some unknown reason.
	As a result skip request triggered by the accept event method stays within
	the critical synchronization region (semaphore synchroMutualExclusion)
	and consequently locks any other activity.
	I have to dig up this problem someday..."

	"
	.
	Transcript show: 'Request for forks ' , (self arg: 1) printString , ' and ' , (self arg: 2) printString , ' by ' , (self sender) printString
					, ' has been skipped ' , numberTimesSkipped printString , ' times.';cr
	"! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SkippedInvocation class
	instanceVariableNames: ''!


!SkippedInvocation class methodsFor: 'instance creation'!

new
	^super new initialize! !

InvocationActivity subclass: #InvocationTableActivity
	instanceVariableNames: 'forksAvailability '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
InvocationTableActivity comment:
'Class InvocationTableActivity implements synchronization of fork requests by philosophers.
Note that availability of each fork is specified as an instance variable of this class.
Updating of the availability is performed at acceptance time.'!


!InvocationTableActivity methodsFor: 'initialize'!

privateInitialize
	"Create and initialize the array containing the availability of each fork."

	super privateInitialize.
	forksAvailability := Array new: bself n withAll: true! !

!InvocationTableActivity methodsFor: 'guards'!

guardOFpickUp: i and: j by: philosopher
	"Both forks must be available."

	^(forksAvailability at: i) & (forksAvailability at: j)!

guardOFputDown: i and: j by: philosopher
	"Always free to release forks."

	^true!

guardOFstart
	^true! !

!InvocationTableActivity methodsFor: 'synchro events'!

synchroEventAccept: anInvocation
	"When accepting a pick up or put down forks invocation, switch the availability of the two requested forks."

	super synchroEventAccept: anInvocation.
	(#(pickUp:and:by: putDown:and:by:) includes: anInvocation selector)
		ifTrue: [self toggleForks: (anInvocation arg: 1) and: (anInvocation arg: 2)]! !

!InvocationTableActivity methodsFor: 'forks availability'!

toggleForks: i and: j
	"Switch availability of the given forks."

	forksAvailability
		at: i put: (forksAvailability at: i) not;
		at: j put: (forksAvailability at: j) not! !

Philosopher subclass: #AtomicPhilosopher
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
AtomicPhilosopher comment:
'Class AtomicPhilosopher is defined as a subclass of class Philosopher.
It issues eatWith:and: requests to the table (rather than pickUp:and: and putDown:and:) as associated table activity class PlainInvocationTableActivity specifies.
(Therefore we name this class AtomicPhilosopher as it atomically requests forks, eats and releases them!!).'!


!AtomicPhilosopher methodsFor: 'body'!

body
	"Body: a philosopher does numberSteps times his life steps, that is: think and eat."
	"Note that this active object class has no script method as it receives no message."

	1 to: numberSteps do: [: i |
		self think.
		Transcript show: self printString , ' wants to eat for the ' , i printString , 'th time and requests forks '
			, leftForkPosition printString , ' and ' , rightForkPosition printString; cr.
		"He requests his forks and eats. Note that he waits (value) for the acknowledgement."
		(table eatWith: leftForkPosition and: rightForkPosition) value.
		Transcript show: self printString , ' has finished eating and releases forks '
			, leftForkPosition printString , ' and ' , rightForkPosition printString; cr]! !

PlainInvocationActivity subclass: #PlainInvocationTableActivity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
PlainInvocationTableActivity comment:
'Class PlainInvocationTableActivity implements an alternative representation of the dining philosophers problem.
Guard of method eatWith:and: specifies that there is no other currently eating philosopher (invocation) using at least one of the requested forks.'!


!PlainInvocationTableActivity methodsFor: 'guards'!

guardOFeatWith: i and: j
	"No other currently eating philosopher (invocation) using at least one of the requested forks."

	^self noCurrent: #eatWith:and: with: [:invocation |
		self areForksSharedBetween: invocation and: currentInvocation]!

guardOFstart
	^true! !

!PlainInvocationTableActivity methodsFor: 'predicates'!

areForksSharedBetween: invocation1 and: invocation2
	"We assume that these are two eatWith:and: invocations
	with left and right fork positions as first and second arguments."

	^((invocation1 arg: 1) = (invocation2 arg: 1))
		or: [((invocation1 arg: 1) = (invocation2 arg: 2))
			or: [(invocation1 arg: 2) = (invocation2 arg: 1)]]! !

PlainInvocationTableActivity subclass: #StarvationFreeTableActivity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
StarvationFreeTableActivity comment:
'Activity class StarvationFreeTableActivity is a subclass of activity class PlainInvocationTableActivity.
It ensures that no request may starve (being indefinitely delayed) by recording the number of times it is skipped (within the invocation instance of class SkippedInvocation) and ensuring that no pending request for eating is skipped more than a certain amount of times.'!


!StarvationFreeTableActivity methodsFor: 'guards'!

guardOFeatWith: i and: j
	"Add the extra constraint that there cannot be another pending eating activation
	which has already been skipped more than two times."

	^(super guardOFeatWith: i and: j)
		and: [self
				noPending: #eatWith:and:
				priorTo: currentInvocation arrivalTime
				andWith: [:invocation |
							(self areForksSharedBetween: invocation and: currentInvocation)
								and: [invocation numberTimesSkipped > 2]]]! !

!StarvationFreeTableActivity methodsFor: 'synchro events'!

synchroEventAccept: anInvocation
	"When accepting one eating invocation, signal all pending activations with a prior arrival time
	and sharing forks with current one that they have been skipped."

	super synchroEventAccept: anInvocation.
	anInvocation ifSelector: #eatWith:and: do:
		[self forAllPending: #eatWith:and: do: [:invocation |
			(invocation arrivalTime < anInvocation arrivalTime
				and: [self areForksSharedBetween: invocation and: anInvocation])
					ifTrue: [invocation skip]]]! !

!StarvationFreeTableActivity methodsFor: 'invocation class'!

invocationClassFor: aMessage
	^aMessage selector = #eatWith:and:
		ifTrue: [SkippedInvocation]
		ifFalse: [super invocationClassFor: aMessage]! !

ImplicitReplyObject subclass: #FCFSPrinter
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
FCFSPrinter comment:
'Class FCFSPrinter implements the example of a printer server with assumption of message preservation ordering (FCFS stands for first come first served).
See its associated activity class FCFSPrinterActivity and its subclasses ShortestJobFirstPrinterActivity and StarvationFreeShortestJobFirstPrinterActivity.'!


!FCFSPrinter methodsFor: 'script'!

print: job length: length
	Transcript show: 'Print job: ' , job printString , ' of length: ' , length printString , ' started '; cr.
	(Delay forSeconds: length) wait.
	Transcript show: 'done!!'; cr! !

!FCFSPrinter methodsFor: 'default classes'!

activityClass
	"Synchronization is specified by the activity guards specification."

	^FCFSPrinterActivity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

FCFSPrinter class
	instanceVariableNames: ''!


!FCFSPrinter class methodsFor: 'example'!

example
	"self example"

	self new active
		print: 1 length: 4;
		print: 2 length: 2;
		print: 3 length: 3!

shortestJobFirstExample
	"self shortestJobFirstExample"

	(self new active: ShortestJobFirstPrinterActivity)
		print: 1 length: 4;
		print: 2 length: 2;
		print: 3 length: 3!

starvationFreeShortestJobFirstExample
	"self starvationFreeShortestJobFirstExample"

	(self new active: StarvationFreeShortestJobFirstPrinterActivity)
		print: 1 length: 4;
		print: 2 length: 2;
		print: 3 length: 3;
		print: 4 length: 1! !

InvocationActivity subclass: #DiskHeadSchedulerActivity
	instanceVariableNames: ''
	classVariableNames: 'NumberOfTracks '
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!


!DiskHeadSchedulerActivity methodsFor: 'initialize'!

privateInitialize
	"Initialize the number of tracks of the Disk."

	super privateInitialize.
	NumberOfTracks := bself numberOfTracks! !

!DiskHeadSchedulerActivity methodsFor: 'guards'!

guardOFmoveHeadTo: destinationTrackNumber andTransfer: data
	^self noCurrent: #moveHeadTo:andTransfer:! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

DiskHeadSchedulerActivity class
	instanceVariableNames: ''!


!DiskHeadSchedulerActivity class methodsFor: 'initialize'!

initialize
	NumberOfTracks := 100! !

DiskHeadSchedulerActivity subclass: #NearestJobNextDiskHeadSchedulerActivity
	instanceVariableNames: 'currentTrackNumber '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!


!NearestJobNextDiskHeadSchedulerActivity methodsFor: 'initialize'!

privateInitialize
	super privateInitialize.
	currentTrackNumber := 0! !

!NearestJobNextDiskHeadSchedulerActivity methodsFor: 'guards'!

guardOFmoveHeadTo: destinationTrackNumber andTransfer: data
	^(super guardOFmoveHeadTo: destinationTrackNumber andTransfer: data)
		and: [self noPending: #moveHeadTo:andTransfer: with: [:invocation |
				(self distanceTo: (invocation arg: 1)) < (self distanceTo: (currentInvocation arg: 1))]]! !

!NearestJobNextDiskHeadSchedulerActivity methodsFor: 'synchro events'!

synchroEventComplete: anInvocation
	"Records new current track number, once the head has been moved."

	super synchroEventComplete: anInvocation.
	anInvocation ifSelector: #moveHeadTo:andTransfer: do:
		[currentTrackNumber := anInvocation arg: 1]! !

!NearestJobNextDiskHeadSchedulerActivity methodsFor: 'private'!

distanceTo: destinationTrackNumber
	^(destinationTrackNumber - currentTrackNumber) abs! !

NearestJobNextDiskHeadSchedulerActivity subclass: #ElevatorDiskHeadSchedulerActivity
	instanceVariableNames: 'isGoingUp '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!


!ElevatorDiskHeadSchedulerActivity methodsFor: 'initialize'!

privateInitialize
	super privateInitialize.
	isGoingUp := true! !

!ElevatorDiskHeadSchedulerActivity methodsFor: 'synchro events'!

synchroEventAccept: anInvocation
	"Records if current direction is going to change."

	| destinationTrackNumber |
	super synchroEventAccept: anInvocation.
	anInvocation ifSelector: #moveHeadTo:andTransfer: do:
		[destinationTrackNumber := anInvocation arg: 1.
		destinationTrackNumber < currentTrackNumber
			ifTrue: [isGoingUp := false]
			ifFalse: [destinationTrackNumber > currentTrackNumber
						ifTrue: [isGoingUp := true]]]! !

!ElevatorDiskHeadSchedulerActivity methodsFor: 'private'!

distanceTo: destinationTrackNumber
	"Computes distance between current and possible next position, weighted by possible change of direction."

	destinationTrackNumber = currentTrackNumber
		ifTrue: [^0].
	isGoingUp & (destinationTrackNumber > currentTrackNumber)
		ifTrue: [^destinationTrackNumber - currentTrackNumber].
	(isGoingUp not) & (destinationTrackNumber < currentTrackNumber)
		ifTrue: [^currentTrackNumber - destinationTrackNumber].
	isGoingUp & (destinationTrackNumber < currentTrackNumber)
		ifTrue: [^NumberOfTracks * 2 - destinationTrackNumber - currentTrackNumber].
	(isGoingUp not) & (destinationTrackNumber > currentTrackNumber)
		ifTrue: [^currentTrackNumber + destinationTrackNumber].
	self error: 'Strange case happens...'! !

ElevatorDiskHeadSchedulerActivity subclass: #OptimizedElevatorDiskHeadSchedulerActivity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!


!OptimizedElevatorDiskHeadSchedulerActivity methodsFor: 'guards'!

guardOFmoveHeadTo: destinationTrackNumber andTransfer: data
	^(self noCurrent: #moveHeadTo:andTransfer:)
		and: [self noPending: #moveHeadTo:andTransfer: with: [:invocation |
				invocation distance < currentInvocation distance]]! !

!OptimizedElevatorDiskHeadSchedulerActivity methodsFor: 'synchro events'!

synchroEventAccept: anInvocation
	"Maintain/update distance information of pending invocations."

	| deltaHeadMovement |
	anInvocation ifSelector: #moveHeadTo:andTransfer: do:
		[deltaHeadMovement := self distanceTo: (anInvocation arg: 1).
		self forAllPending: #moveHeadTo:andTransfer: do: [:invocation |
			invocation distance: invocation distance - deltaHeadMovement]].
	super synchroEventAccept: anInvocation!

synchroEventReceive: anInvocation
	"Compute and assign the distance of a distance invocation."

	super synchroEventReceive: anInvocation.
	anInvocation ifSelector: #moveHeadTo:andTransfer: do:
		[anInvocation distance: (self distanceTo: (anInvocation arg: 1))]! !

!OptimizedElevatorDiskHeadSchedulerActivity methodsFor: 'invocation class'!

invocationClassFor: aMessage
	^aMessage selector = #moveHeadTo:andTransfer:
		ifTrue: [DistanceInvocation]
		ifFalse: [super invocationClassFor: aMessage]! !

StandardInvocation subclass: #DistanceInvocation
	instanceVariableNames: 'distance '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!


!DistanceInvocation methodsFor: 'accessing'!

distance
	^distance!

distance: anInteger
	distance := anInteger! !

PlainInvocationObject subclass: #PlainInvocationTable
	instanceVariableNames: 'n p numberSteps philosophers '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
PlainInvocationTable comment:
'Class PlainInvocationTable implements an alternative representation of the dining philosophers problem.
As opposed to class InvocationTable, availability of forks is not represented as a table (set of variables). By being able to access to the set of current invocations (which is managed by class PlainActivity) we specify the guard as checking that no fork should be shared by any current invocation.
Methods pickUp:and: and putDown:and: are replaced by one method eatWith:and:.
We need to define a new class of philosophers (AtomicPhilosopher) to issue eatWith:and: requests to the table.

See associated activity class PlainInvocationTableActivity.
This solution has been initially described in the report on Synchronization Variables (see class InvocationActivity comment).

Note that PlainInvocationTableActivity is defined as a subclass of behavior class PlainInvocationObject because it makes use of the sender method to access current sender (from within method eatWith:and:).'!


!PlainInvocationTable methodsFor: 'initialize'!

initializeNumberForks: numberForks numberPhilosophersPerFork: numberPhilosophersPerFork numberSteps: steps
	n := numberForks.
	p := numberPhilosophersPerFork.
	numberSteps := steps!

makePhilosophers
	"Create and initialize the n * p philosophers.
	There are p philosophers for each position around the table."

	philosophers := Array new: n*p.
	1 to: n*p do:
		[:i | philosophers at: i put:
			(AtomicPhilosopher new
				number: i
				position: i - 1 \\ n + 1
				n: n
				table: aself
				numberSteps: numberSteps)
					active]! !

!PlainInvocationTable methodsFor: 'script'!

eatWith: i and: j
	"Empty body (only tracing and random sleeping), only synchronization."

	| delay |
	delay := (Random new next * 5) rounded + 1.
	Transcript show: self sender printString , ' eats with forks '
			, i printString , ' and ' , j printString
			, ' during ' , delay printString , ' seconds.'; cr.
	(Delay forSeconds: delay) wait!

start
	self makePhilosophers! !

!PlainInvocationTable methodsFor: 'default classes'!

activityClass
	"Synchronization is specified by the activity guards specification."

	^PlainInvocationTableActivity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

PlainInvocationTable class
	instanceVariableNames: ''!


!PlainInvocationTable class methodsFor: 'example'!

example: n
	"self example: 5"

	(self new initializeNumberForks: n numberPhilosophersPerFork: 2 numberSteps: 3) active start!

starvationFreeExample: n
	"self starvationFreeExample: 5"

	((self new initializeNumberForks: n numberPhilosophersPerFork: 2 numberSteps: 3)
		active: StarvationFreeTableActivity) start! !

GuardsTable subclass: #InvocationTable
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
InvocationTable comment:
'Class InvocationTable implements the dining philosophers example with guards and generic invocations.
As opposed to class GuardsTable, note that availability of each fork is specified by the associated activity class and not by this class. Checking and updating fork availability is a pure synchronization problem.
See associated activity class InvocationTableActivity.

This solution has been initially described in the report on Synchronization Variables (see class InvocationActivity comment).'!


!InvocationTable methodsFor: 'initialize'!

makeForksAvailability
	"Do nothing because it is managed by the activity class."! !

!InvocationTable methodsFor: 'accessing'!

n
	^n! !

!InvocationTable methodsFor: 'script'!

pickUp: i and: j by: philosopher
	"Empty, only synchronization."!

putDown: i and: j by: philosopher
	"Empty, only synchronization."! !

!InvocationTable methodsFor: 'default classes'!

activityClass
	"Synchronization is specified by the activity guards specification."

	^InvocationTableActivity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

InvocationTable class
	instanceVariableNames: ''!


!InvocationTable class methodsFor: 'example'!

example: n
	"self example: 5"

	(self new initializeNumberForks: n numberPhilosophersPerFork: 2 numberSteps: 3) active start! !

InvocationActivity subclass: #AlarmClockActivity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
AlarmClockActivity comment:
'Activity class AlarmClockActivity implements the alarm clock problem.
It computes and assigns the wake up time into the invocation of the wake up request (invocation class WakeUpTimeInvocation).
The guard for wake up checks if current time is equal or greater than wake up time.'!


!AlarmClockActivity methodsFor: 'guards'!

guardOFtick
	^true!

guardOFwakeUpAfter: time
	^self currentTime >= currentInvocation wakeUpTime! !

!AlarmClockActivity methodsFor: 'accessing'!

currentTime
	"Current time is represented by the number of completed ticks."

	^self completed: #tick! !

!AlarmClockActivity methodsFor: 'synchro events'!

synchroEventReceive: anInvocation
	"Set up the waking time of a wake up invocation."

	super synchroEventReceive: anInvocation.
	anInvocation ifSelector: #wakeUpAfter: do:
		[Transcript show: self printString , ' sets wake up time ' , (anInvocation arg: 1) printString , ' ticks later.'; cr.
		anInvocation wakeUpTime: self currentTime + (anInvocation arg: 1)]! !

!AlarmClockActivity methodsFor: 'invocation class'!

invocationClassFor: aMessage
	^aMessage selector = #wakeUpAfter:
		ifTrue: [WakeUpTimeInvocation]
		ifFalse: [super invocationClassFor: aMessage]! !

InvocationActivity subclass: #FCFSPrinterActivity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
FCFSPrinterActivity comment:
'Activity class FCFSPrinterActivity specifies the synchronization of its associated behavior class, that is ensures serialization and message ordering preservation (first come first served).'!


!FCFSPrinterActivity methodsFor: 'guards'!

guardOFprint: job length: length
	"One job at a time and no prior invocation still pending."

	^(self noCurrent: #print:length:)
		and: [self noPending: #print:length: priorTo: currentInvocation arrivalTime]! !

FCFSPrinterActivity subclass: #ShortestJobFirstPrinterActivity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
ShortestJobFirstPrinterActivity comment:
'Activity class ShortestJobFirstPrinterActivity is a subclass of activity class FCFSPrinterActivity.
It gives priority to the shortest job.'!


!ShortestJobFirstPrinterActivity methodsFor: 'guards'!

guardOFprint: job length: length
	"One printing job at a time and no pending print invocation with a shorter job size
	or with the same size but with some prior arrival time."

	^(self noCurrent: #print:length:)
		and: [self noPending: #print:length: with: [:invocation |
				invocation jobSize < currentInvocation jobSize
					or: [invocation jobSize = currentInvocation jobSize
						and: [invocation arrivalTime < currentInvocation arrivalTime]]]]! !

!ShortestJobFirstPrinterActivity methodsFor: 'synchro events'!

synchroEventReceive: anInvocation
	"Set up the job size of printing invocations.
	In this case simply copies the second argument being the length.
	There may be some general computation prediction taking place here."

	super synchroEventReceive: anInvocation.
	anInvocation ifSelector: #print:length: do:
		[anInvocation jobSize: (anInvocation arg: 2)]! !

!ShortestJobFirstPrinterActivity methodsFor: 'invocation class'!

invocationClassFor: aMessage
	^aMessage selector = #print:length:
		ifTrue: [JobSizeInvocation]
		ifFalse: [super invocationClassFor: aMessage]! !

ShortestJobFirstPrinterActivity subclass: #StarvationFreeShortestJobFirstPrinterActivity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
StarvationFreeShortestJobFirstPrinterActivity comment:
'Activity class StarvationFreeShortestJobFirstPrinterActivity is a subclass of activity class ShortestJobFirstPrinterActivity.
It ensures that no request may starve (being indefinitely delayed) by decreasing its job size each time it has been skipped.
It uses invocation class JobSizeInvocation.'!


!StarvationFreeShortestJobFirstPrinterActivity methodsFor: 'synchro events'!

synchroEventAccept: anInvocation
	"When accepting one printing invocation,
	decrease job size of all pending activations with a prior arrival time."
	"This ensures that there won't be starvation."

	super synchroEventAccept: anInvocation.
	anInvocation ifSelector: #print:length: do:
		[self forAllPending: #print:length: do: [:invocation |
			invocation arrivalTime < anInvocation arrivalTime
				ifTrue: [invocation decrJobSize]]]! !

StandardInvocation subclass: #WakeUpTimeInvocation
	instanceVariableNames: 'wakeUpTime '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
WakeUpTimeInvocation comment:
'Invocation class WakeUpTimeInvocation adds an instance variable (wakeUpTime) to record wake up time which is set at reception time.
It is used by activity class AlarmClockActivity.'!


!WakeUpTimeInvocation methodsFor: 'accessing'!

wakeUpTime
	^wakeUpTime!

wakeUpTime: time
	wakeUpTime := time! !

ImplicitReplyObject subclass: #PriorityPrinter
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
PriorityPrinter comment:
'Class PriorityPrinter implements the example of the dynamic priority printer. (Each group of users gets assigned some priority. This priority may be reassigned dynamically).
Note that method updatePriorityOf:to: has no method body. It is pure synchronization.
See associated activity class PriorityPrinterActivity.

This solution has been initially described in the report on Synchronization Variables (see class InvocationActivity comment).'!


!PriorityPrinter methodsFor: 'script'!

print: job length: length group: group
	Transcript show: 'Print job: ' , job printString , ' of length: ' , length printString , ' and group: ' , group printString , ' started '; cr.
	(Delay forSeconds: length) wait.
	Transcript show: 'done!!'; cr!

updatePriorityOf: group to: newPriority
	"Empty, only synchronization code."

	Transcript show: 'Priority of group ' , group asString , ' updated to ' , newPriority printString; cr! !

!PriorityPrinter methodsFor: 'default classes'!

activityClass
	"Synchronization is specified by the activity guards specification."

	^PriorityPrinterActivity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

PriorityPrinter class
	instanceVariableNames: ''!


!PriorityPrinter class methodsFor: 'example'!

exampleWithDynamicPriorityChange: isDynamic
	"self exampleWithDynamicPriorityChange: false"
	"self exampleWithDynamicPriorityChange: true"

	"Compare ordering of requests with a fixed priority and with a dynamic change."

	| printer |
	printer := self new active.
	printer
		"First initialize the priority of each current group."
		updatePriorityOf: #ungrad to: 1;
		updatePriorityOf: #grad to: 2;
		updatePriorityOf: #staff to: 3;
		"Send printing requests."
		print: 1 length: 1 group: #ungrad;
		print: 2 length: 2 group: #staff;
		print: 3 length: 3 group: #ungrad;
		print: 4 length: 1 group: #ungrad;
		print: 5 length: 1 group: #staff.
	isDynamic ifTrue:
		[(Delay forSeconds: 2) wait.
		"Temporarily (fools day!!) lowest priority group gets the highest of all."
		printer updatePriorityOf: #ungrad to: 4.
		(Delay forSeconds: 5) wait.
		"Fools day is over. Return to standard policy."
		printer updatePriorityOf: #ungrad to: 1]! !

StandardInvocation subclass: #JobSizeInvocation
	instanceVariableNames: 'jobSize '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation-Ex'!
JobSizeInvocation comment:
'Invocation class JobSizeInvocation adds an instance variable (jobSize) to record the size of the job.
This is useful in a starvation free policy where the job size may be decreased each time one invocation is skipped by being prefered a shorter size job.
See activity class StarvationFreeShortestJobFirstPrinterActivity.'!


!JobSizeInvocation methodsFor: 'accessing'!

decrJobSize
	jobSize := jobSize - 1!

jobSize
	^jobSize!

jobSize: anInteger
	jobSize := anInteger! !
DiskHeadSchedulerActivity initialize!

DiskHeadScheduler initialize!

