WithReplyDestInvocation subclass: #StandardInvocation
	instanceVariableNames: 'arrivalTime sender '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation'!
StandardInvocation comment:
'Class StandardInvocation is a subclass of class WithReplyDestInvocation.

It inherits implicit reply destination instance variable (replyDest).
It adds two new variables: arrivalTime and sender.
It also defines accessing method to a given (by its index) argument of the invocation.

Note that unfortunately fetching an argument of a message by its name is uneasy in Smalltalk. There is no direct way to relate names of parameters as defined in a method message pattern with the arguments within a message.'!


!StandardInvocation methodsFor: 'accessing'!

arg: i
	"Accessing to the ith argument of the invocation."

	^args at: i!

arrivalTime
	^arrivalTime!

arrivalTime: anInteger
	arrivalTime := anInteger!

sender
	^sender!

sender: anObject
	sender := anObject! !

!StandardInvocation methodsFor: 'selecting'!

ifSelector: aSelector do: aBlock
	"Useful as a shortcut for filtering along the selector
	 e.g., to perform initialization of specific invocations depending on which message (selector)."

	^selector = aSelector
		ifTrue: aBlock! !

StandardInvocation subclass: #PlainInvocation
	instanceVariableNames: 'process '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation'!
PlainInvocation comment:
'Class PlainInvocation is a subclass of class StandardInvocation.
It adds one instance variable (process) to record the subprocess created to actually compute the invocation.
This may be used in order for a behavior to access current invocation by using this process as an index. See behavior class PlainInvocationObject.'!


!PlainInvocation methodsFor: 'accessing'!

process
	^process!

process: aProcess
	process := aProcess! !

CountersActivity subclass: #InvocationActivity
	instanceVariableNames: 'currentInvocation reevaluationSemaphore '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation'!
InvocationActivity comment:
'Activity class InvocationActivity implements model of guards with synchronization counters with added features in order to provide more expressiveness.

Class InvocationActivity is a subclass of activity class CountersActivity.
It adds the following features:
	reference to current invocation,
	better implementation of reevaluation semantics of guards,
	systematic use of generic activations, extended with arrival time information,
	many constructs to loop over pending activations in order to have greater synchronization expressiveness.

Note that reference to pseudo-variable currentInvocation is safe ONLY within a guard evaluation (as it is within the atomic guard evaluation). But it is NOT safe to be used in some event (actually it is safe within accept synchronization event) because it may have been reassigned by some following accepted invocation.
Note that subclass PlainInvocationActivity provides the means for the behavior (associated behavior class PlainInvocationObject) to also access current activation from within a method.

Class InvocationActivity actually implements most of the model of synchronization as described in the technical report Synchronization Variables, Ciara McHale et al., TCD-CS-94-01, University of Dublin, January 1994.

Instance variables:

	currentInvocation			<Invocation>		current invocation being checked for acceptance.
	reevaluationSemaphore	<Semaphore>	controls the reevaluation of guards. Guards of pending invocations won''t be reevaluated until some event has occured.

See examples in category Actalk-Ext-Invocation-Ex to describe various problems of synchronization involving time ordering, priorities, starvation avoidance...'!


!InvocationActivity methodsFor: 'initialize'!

privateInitialize
	super privateInitialize.
	reevaluationSemaphore := Semaphore new! !

!InvocationActivity methodsFor: 'activity setting'!

body
	"Define the behavior body."

	| i mailBox|
	mailBox := self mailBox.
	"Infinitely do the following cycle."
	[true] whileTrue:
		["Initialize the index i for accessing messages."
		i := 1.
		"Successively for each message in the mailBox."
		[i <= (mailBox size)] whileTrue:
			[currentInvocation := mailBox basicAtIndex: i.
			"Check if the message may be accepted."
			(self isCandidateMessage: currentInvocation)
				ifTrue:
					["If the case, then remove it from the mailbox and then accept it."
					mailBox removeAtIndex: i.
					self acceptMessage: currentInvocation
					"Note that we don't increment the index as we removed that message."]
				ifFalse:
					["Increment the index in order to check next message."
					i := i + 1]].
	"Once we checked all messages within the mailbox, we wait for any event (receive, accept, complete)
	to occur or if it has already occured.
	We then remove any excess signal to resume the cycle with a semaphore refreshed with no prior event occurence."
	reevaluationSemaphore wait; initSignals]! !

!InvocationActivity methodsFor: 'synchro events'!

synchroEventAccept: anInvocation
	"Signal the guards reevaluation semaphore as some event occurs."

	super synchroEventAccept: anInvocation.
	reevaluationSemaphore signal!

synchroEventComplete: anInvocation
	"Signal the guards reevaluation semaphore as some event occurs."

	super synchroEventComplete: anInvocation.
	reevaluationSemaphore signal!

synchroEventReceive: anInvocation
	"Assign a new time stamp to the invocation."
	"Signal the guards reevaluation semaphore as some event occurs."

	super synchroEventReceive: anInvocation.
	anInvocation arrivalTime: address nextTimeStamp.
	reevaluationSemaphore signal! !

!InvocationActivity methodsFor: 'enumerating invocations'!

forAllPending: selector do: aBlock
	"For all pending invocations of selector evaluate aBlock.
	aBlock has one argument being the pending invocation."

	^self forAllPendingDo: [:invocation |
		invocation selector = selector
			ifTrue: [aBlock value: invocation]]!

forAllPendingDo: aBlock
	"For all pending invocation evaluate aBlock.
	aBlock has one argument being the pending invocation."

	^self mailBox do: aBlock! !

!InvocationActivity methodsFor: 'predicates on invocations'!

noCurrent: selector
	"Check if there is no current invocation of selector.
	Return a boolean."

	^(self current: selector) = 0!

noPending: selector
	"Check if there is no pending invocation of selector.
	Return a boolean."

	^(self pending: selector) = 0!

noPending: selector priorTo: arrivalTime
	"Check if there is no prior (arrived earlier) pending invocation of selector.
	That is ensure message ordering preservation for selector.
	Note: departure time (from the sender) is not considered.
	Return a boolean."

	^self noPending: selector priorTo: arrivalTime andWith: [:invocation | true]!

noPending: selector priorTo: arrivalTime andWith: aBlock
	"Check if there is no prior pending invocation of selector which satisfies condition aBlock.
	aBlock has one argument being the pending invocation.
	Note: departure time (from the sender) is not considered.
	Return a boolean."

	^self noPending: selector with: [:invocation |
		invocation arrivalTime < arrivalTime
			and: [aBlock value: invocation]]!

noPending: selector with: aBlock
	"Check if there is no pending invocation of selector which satisfies condition aBlock.
	aBlock has one argument being the pending invocation.
	Return a boolean."

	^self noPendingWith: [:invocation |
		invocation selector = selector
			and: [aBlock value: invocation]]!

noPendingPriorTo: arrivalTime
	"Check if there is no prior (arrived earlier) pending invocation which may be accepted
	(that is satisfies its guard).
	That is ensure (global) message ordering preservation.
	Note: departure time (from the sender) is not considered.
	Return a boolean."

	^self noPendingWith: [:invocation |
		invocation arrivalTime < arrivalTime
			and: [self evaluateGuardForMessage: invocation]]!

noPendingWith: aBlock
	"Check if there is no pending invocation which satisfies condition aBlock.
	aBlock has one argument being the pending invocation.
	Return a boolean."

	^(self mailBox detect: aBlock ifNone: [nil]) isNil! !

!InvocationActivity methodsFor: 'default classes'!

addressClass
	"InvocationActivity
		is defined in order to dispatch onto invocation class PlainInvocation
		which should be initialized by
	InvocationAddress method
		receiveMessage:."

	^InvocationAddress!

invocationClassFor: aMessage
	"Standard invocation recording implicit reply destination, arrival time and sender."

	^StandardInvocation! !

!InvocationActivity methodsFor: 'compatibility constraints'!

addressConstraint
	"Ensure constraint defined by method addressClass."

	^InvocationAddress! !

SynchroConcurrentAddress subclass: #InvocationAddress
	instanceVariableNames: 'arrivalCounter '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation'!
InvocationAddress comment:
'Address class InvocationAddress implements time stamp to record arrival time of messages. It also record the sender of the message and triggers generic event method for sending, as for address class WithSenderClass.
It is defined as a subclass of address class SynchroConcurrentAddress.

Instance variables:

	arrivalCounter	<Integer>	records the number of messages received and is used as a time stamp.'!


!InvocationAddress methodsFor: 'initialize'!

privateInitialize
	super privateInitialize.
	arrivalCounter := 0! !

!InvocationAddress methodsFor: 'accessing'!

nextTimeStamp
	"Increment and return next time stamp showing arrival order."

	^arrivalCounter := arrivalCounter + 1! !

!InvocationAddress methodsFor: 'message passing'!

receiveMessage: aMessage
	"Receive an extended invocation with the sender object address (reference).
	Before that, trigger the generic event for the sender to send a message."

	| sender |
	sender := thisContext sender sender homeReceiver address.
	sender isNil ifFalse:
		[sender asObject eventSend: aMessage to: self].
	^self receiveGenericMessage:
		((activity invocationClassFor: aMessage) new
				setSelector: aMessage selector arguments: aMessage arguments;
				sender: sender)! !

!InvocationAddress methodsFor: 'compatibility constraints'!

activityConstraint
	"InvocationAddress method
		receiveMessage:
			assumes that the activity returns an invocation instance of class StandardInvocation
			(at least having a slot for sender)
			as defined by activity class
	InvocationActivity."

	^InvocationActivity! !

InvocationActivity subclass: #PlainInvocationActivity
	instanceVariableNames: 'currentInvocations completedInvocations '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation'!
PlainInvocationActivity comment:
'Activity class PlainInvocationActivity is a subclass of class InvocationActivity.
It records current invocations (currently being computed) and completed invocations.
This gives more expressiveness at some extra cost for this management. Note also that as all completed invocations are recorded, this set has no size boundary unless it gets flushed.

Instance variables:

	currentInvocations		<OrderedCollection>	collection of current invocations,
	completedInvocations		<OrderedCollection>	collection of completed invocations.

See example of class PlainInvocationTable in category Actalk-Ext-Invocation-Ex as an example of describing the dining philosophers problem with no record of fork status but through ensuring that no fork is shared between concurrent eating computations.'!


!PlainInvocationActivity methodsFor: 'initialize'!

privateInitialize
	super privateInitialize.
	currentInvocations := OrderedCollection new.
	completedInvocations := OrderedCollection new! !

!PlainInvocationActivity methodsFor: 'accessing'!

currentInvocations
	"This accessing method is useful for the behavior to access to current activation.
	See method currentActivation of class PlainInvocationObject."
	"From within the activity process, one may use instance variable currentInvocation
	from within a guard method."

	^currentInvocations! !

!PlainInvocationActivity methodsFor: 'activity setting'!

startSubProcessFor: anInvocation
	"Create and start a new subprocess to consume a message."
	"Record the subprocess consuming the invocation within the process slot of the invocation.
	The subprocess may be used as an index to retrieve current invocation.
	See method currentInvocation of associated behavior class PlainInvocationObject."

	| subprocess |
	subprocess := [self subProcessBodyFor: anInvocation] newProcess.
	anInvocation process: subprocess.
	subprocess resume! !

!PlainInvocationActivity methodsFor: 'synchro events'!

synchroEventAccept: anInvocation
	"Add the invocation being accepted in the set of current activations."
	"Note that it occurs BEFORE the synchronization event."

	currentInvocations addLast: anInvocation.
	super synchroEventAccept: anInvocation!

synchroEventComplete: anInvocation
	"Transfer the invocation into the set of completed activations."
	"Note that it occurs AFTER the synchronization event."
	"It removes the reference to the subprocess which is terminated so that it may be garbaged."

	super synchroEventComplete: anInvocation.
	currentInvocations remove: anInvocation.
	completedInvocations addLast: anInvocation.
	anInvocation process: nil! !

!PlainInvocationActivity methodsFor: 'enumerating invocations'!

forAllCompleted: selector do: aBlock
	"For all completed invocations of selector evaluate aBlock.
	aBlock has one argument being the completed invocation."

	^self forAllCompletedDo: [:invocation |
		invocation selector = selector
			ifTrue: [aBlock value: invocation]]!

forAllCompletedDo: aBlock
	"For all completed invocations evaluate aBlock.
	aBlock has one argument being the completed invocation."

	^completedInvocations do: aBlock!

forAllCurrent: selector do: aBlock
	"For all current invocations of selector evaluate aBlock.
	aBlock has one argument being the current invocation."

	^self forAllCurrentDo: [:invocation |
		invocation selector = selector
			ifTrue: [aBlock value: invocation]]!

forAllCurrentDo: aBlock
	"For all current invocations evaluate aBlock.
	aBlock has one argument being the current invocation."

	^currentInvocations do: aBlock! !

!PlainInvocationActivity methodsFor: 'predicates on invovations'!

noCompleted: selector with: aBlock
	"Check if there is no completed invocation of selector which satisfies condition aBlock.
	aBlock has one argument being the completed invocation.
	Return a boolean."

	^self noCompletedWith: [:invocation |
		invocation selector = selector
			and: [aBlock value: invocation]]!

noCompletedWith: aBlock
	"Check if there is no completed invocation which satisfies condition aBlock.
	aBlock has one argument being the completed invocation.
	Return a boolean."

	^(completedInvocations detect: aBlock ifNone: [nil]) isNil!

noCurrent: selector with: aBlock
	"Check if there is no current invocation of selector which satisfies condition aBlock.
	aBlock has one argument being the current invocation.
	Return a boolean."

	^self noCurrentWith: [:invocation |
		invocation selector = selector
			and: [aBlock value: invocation]]!

noCurrentWith: aBlock
	"Check if there is no current invocation which satisfies condition aBlock.
	aBlock has one argument being the current invocation.
	Return a boolean."

	^(currentInvocations detect: aBlock ifNone: [nil]) isNil! !

!PlainInvocationActivity methodsFor: 'reset'!

flushCompletedInvocations
	"Flush the record of completed invocations."
	"Useful to free up used memory when not needed any more."

	^completedInvocations := OrderedCollection new! !

!PlainInvocationActivity methodsFor: 'default classes'!

invocationClassFor: aMessage
	"Plain invocation adds to standard invocation record of the process computing it."

	^PlainInvocation! !

ImplicitReplyObject subclass: #PlainInvocationObject
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Invocation'!
PlainInvocationObject comment:
'Class PlainInvocationObject provides access to current invocation by using the subprocess as an index.
It is defined as a subclass of behavior class ImplicitReplyObject.'!


!PlainInvocationObject methodsFor: 'invocation accessing'!

currentInvocation
	"Return current invocation."
	"This relies on the indexing of current invocations by the subprocess which actually computes them.
	Using current active process as an index, we may find out the right current activation."

	^self activity currentInvocations
			detect: [:invocation |
					invocation process = Processor activeProcess]
			ifNone: [self error: 'cannot find current invocation']!

sender
	"Return sender of current invocation."

	^self currentInvocation sender! !

!PlainInvocationObject methodsFor: 'default classes'!

activityClass
	"PlainInvocationObject method
		currentInvocation
			calls
	PlainInvocationActivity method
		currentInvocations."

	^PlainInvocationActivity! !

!PlainInvocationObject methodsFor: 'compatibility constraints'!

activityConstraint
	"Ensure constraint defined by method activityClass."

	^PlainInvocationActivity! !

