ConcurrentSmalltalkObject subclass: #ConcurrentSmalltalkBoundedBuffer
	instanceVariableNames: 'array getIndex putIndex maxSize '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Suspend-Ex'!
ConcurrentSmalltalkBoundedBuffer comment:
'Class ConcurrentSmalltalkBoundedBuffer implements the bounded buffer example with method suspension and post actions.
In order to motivate use of post actions (for the get method) we follow a model of representation with explicit indexes, as for class ConcurrentBoundedBuffer.'!


!ConcurrentSmalltalkBoundedBuffer methodsFor: 'initialize'!

initialize: anInteger
	maxSize := anInteger.
	array := Array new: maxSize + 1.
	getIndex := 1.
	putIndex := 1! !

!ConcurrentSmalltalkBoundedBuffer methodsFor: 'script'!

get
	"Wait until buffer is not empty."

	self waitUntil: [self isEmpty not].
	self post:
		["Reinitialization occurs after returning the item."
		getIndex := (getIndex \\ array size) + 1.
		(Delay forSeconds: 2) wait].
	^array at: getIndex!

put: item
	"Wait until buffer is not full."

	self waitUntil: [self isFull not].
	array at: putIndex put: item.
	putIndex := (putIndex \\ array size) + 1.
	(Delay forSeconds: 2) wait! !

!ConcurrentSmalltalkBoundedBuffer methodsFor: 'state predicates'!

isEmpty
	^getIndex = putIndex!

isFull
	^(putIndex \\ array size) = (getIndex - 1)! !

!ConcurrentSmalltalkBoundedBuffer methodsFor: 'printing'!

printOn: aStream
	super printOn: aStream.
	aStream nextPut: $(.
	getIndex <= putIndex
		ifTrue:
			[getIndex to: putIndex - 1 do:
				[:i |
				(array at: i) printOn: aStream.
				aStream nextPut: $ ]]
		ifFalse:
			[getIndex to: array size do:
				[:i |
				(array at: i) printOn: aStream.
				aStream nextPut: $ ].
			1 to: putIndex -1 do:
				[:i |
				(array at: i) printOn: aStream.
				aStream nextPut: $ ]].
	aStream nextPut: $)! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ConcurrentSmalltalkBoundedBuffer class
	instanceVariableNames: ''!


!ConcurrentSmalltalkBoundedBuffer class methodsFor: 'instance creation'!

new: size
	^self new initialize: size! !

!ConcurrentSmalltalkBoundedBuffer class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGet: numberItems! !

ConcurrentSmalltalkBoundedBuffer subclass: #ConcurrentSmalltalk2BoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Suspend-Ex'!
ConcurrentSmalltalk2BoundedBuffer comment:
'Class ConcurrentSmalltalkBoundedBuffer implements the bounded buffer example with method suspension and post actions.
Post action of method get is defined as a post method.'!


!ConcurrentSmalltalk2BoundedBuffer methodsFor: 'script'!

get
	"Wait until buffer is not empty."

	self waitUntil: [self isEmpty not].
	"Reinitialization is specified in the associated post action method."
	^array at: getIndex!

postOFget
	"Reinitialization of the buffer occurs after returning the item."

	getIndex := (getIndex \\ array size) + 1.
	(Delay forSeconds: 2) wait! !

!ConcurrentSmalltalk2BoundedBuffer methodsFor: 'default classes'!

activityClass
	"Example of using post actions methods."

	^ConcurrentSmalltalk2Activity! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ConcurrentSmalltalk2BoundedBuffer class
	instanceVariableNames: ''!


!ConcurrentSmalltalk2BoundedBuffer class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	super exampleSize: maxSize numberItems: numberItems speedRatio: ratio! !

SuspendObject subclass: #SuspendBoundedBuffer
	instanceVariableNames: 'contents maxSize '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Suspend-Ex'!
SuspendBoundedBuffer comment:
'Class SuspendBoundedBuffer implements the bounded buffer example with method suspension.
Put and get methods suspend as long as their condition (respectively, not full and not empty) is not fullfilled.'!


!SuspendBoundedBuffer methodsFor: 'initialize'!

initialize: anInteger
	maxSize := anInteger.
	contents := BoundedBufferObject new maxSize: maxSize! !

!SuspendBoundedBuffer methodsFor: 'script'!

get
	"Wait until buffer is not empty."

	self waitUntil: [contents isEmpty not].
	^contents removeFirst!

put: item
	"Wait until buffer is not full."

	self waitUntil: [contents isFull not].
	contents addLast: item! !

!SuspendBoundedBuffer methodsFor: 'printing'!

printOn: aStream
	"Print as default followed by its contents."

	super printOn: aStream.
	contents printOn: aStream! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SuspendBoundedBuffer class
	instanceVariableNames: ''!


!SuspendBoundedBuffer class methodsFor: 'instance creation'!

new: size
	^self new initialize: size! !

!SuspendBoundedBuffer class methodsFor: 'example'!

exampleSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleSize: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGet: numberItems! !

SuspendBoundedBuffer subclass: #SuspendGgetBoundedBuffer
	instanceVariableNames: 'afterPut '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Suspend-Ex'!
SuspendGgetBoundedBuffer comment:
'This class implements an example of inheritance anomaly.'!


!SuspendGgetBoundedBuffer methodsFor: 'initialize'!

initialize
	afterPut := false! !

!SuspendGgetBoundedBuffer methodsFor: 'script'!

gget
	"We cannot reuse previous definition of method get.
	Naive definition as:
		self waitUntil: [afterPut not].
		^super get
	is not correct.
	This is because we need to ensure atomically that both conditions (not after a put and then not empty) are true.
	Otherwise when second condition (not empty) is true,
	it could be that first condition (not after a put) is not true anymore!!"

	self waitUntil: [(contents isEmpty not) & (afterPut not)].
	^contents removeFirst! !

!SuspendGgetBoundedBuffer methodsFor: 'events'!

eventAccept: aMessage
	Transcript show: self printString , ' accept (' , aMessage compactPrintString , ')'; cr!

eventComplete: aMessage
	Transcript show: self printString , ' complete (' , aMessage compactPrintString , ')'; cr!

eventReceive: aMessage
	Transcript show: self printString , ' receive (' , aMessage compactPrintString , ')'; cr!

kernelEventComplete: aMessage
	super kernelEventComplete: aMessage.
	afterPut := aMessage selector = #put:! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SuspendGgetBoundedBuffer class
	instanceVariableNames: ''!


!SuspendGgetBoundedBuffer class methodsFor: 'instance creation'!

new
	^super new initialize! !

!SuspendGgetBoundedBuffer class methodsFor: 'example'!

exampleGgetSize: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleGgetSize: 3 numberItems: 8 speedRatio: 3"
	"self exampleGgetSize: 3 numberItems: 8 speedRatio: 1/3"

	"Note that there is no guarantee that all gget requests will be served.
	As a result the bounded buffer may stop before all producer and consumer requests had been served.
	Remind also that we only specify that gget cannot be accepted after COMPLETION of a put.
	Thus a gget could be accepted after acceptance of a put but before its completion if it gets suspended."

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGget: numberItems! !

SuspendBoundedBuffer subclass: #SuspendGet2BoundedBuffer
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Suspend-Ex'!
SuspendGet2BoundedBuffer comment:
'This class implements an example of inheritance anomaly.'!


!SuspendGet2BoundedBuffer methodsFor: 'script'!

get2
	"Wait until buffer contains at least two elements."

	self waitUntil: [contents size >= 2].
	^Array
		with: self get
		with: self get! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SuspendGet2BoundedBuffer class
	instanceVariableNames: ''!


!SuspendGet2BoundedBuffer class methodsFor: 'example'!

exampleGet2Size: maxSize numberItems: numberItems speedRatio: ratio
	"self exampleGet2Size: 3 numberItems: 8 speedRatio: 3"
	"self exampleGet2Size: 3 numberItems: 8 speedRatio: 1/3"

	| buffer producer consumer |
	buffer := (self new: maxSize) active.
	producer := (Producer new buffer: buffer delay: 1) active.
	consumer := (Consumer new buffer: buffer delay: ratio) active.
	producer runPut: numberItems.
	consumer runGet2: numberItems! !

