ImplicitReplyObject subclass: #SuspendObject
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Suspend'!
SuspendObject comment:
'Class SuspendObject is a behavior class which defines the programming construct to suspend computation of a method onto some condition.
This construct, named waitUntil:, is analog to ConcurrentSmalltalk-II relinquish: construct.
Class SuspendObject is associated to activity class SuspendActivity which implements the actual management of suspension of sub activities.'!


!SuspendObject methodsFor: 'waitUntil construct'!

waitUntil: resumptionConditionBlock
	"Suspend current method computation on a resumption condition (block)."
	"Equivalent to relinquish: construct in ConcurrentSmalltalk-II."

	"If resumption condition is true do nothing."
	resumptionConditionBlock value ifFalse:
		[self activity subProcessWaitUntil: resumptionConditionBlock]! !

!SuspendObject methodsFor: 'default classes'!

activityClass
	"SuspendObject method
		waitUntil:
			may call
	SuspendActivity method
		subProcessWaitUntil:."

	^SuspendActivity! !

!SuspendObject methodsFor: 'compatibility constraints'!

activityConstraint
	"Enforce constraint defined by method activityClass."

	^SuspendActivity! !

ConcurrentActivity subclass: #SuspendActivity
	instanceVariableNames: 'suspendedSubProcessAssociationList controlSemaphore '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Suspend'!
SuspendActivity comment:
'Class SuspendActivity implements suspension of methods.
Within a method body, one may express suspension onto some condition.
This is analog to ConcurrentSmalltalk-II relinquish: method.
This is also analog to Kessel monitors except that monitors synchronize external processes, as here processes are internally created for every starting request.

We need to be able to suspend some method computation, and then start another one or restart one previously suspended (and whose suspension condition became false).
Therefore we need a main control activity process. It will act as a scheduler of message/method activations. It launchs suprocesses for computing requests. Then it suspends. It later resumes once a method computation is suspended or completed.
The main control activity process is following.
It first checks if one previously suspended method subprocess is ready to resume (by checking their conditions). If the case, it resumes it. Otherwise it creates a new activity subprocess to process next pending message. (This is a lazy creation as it first waits for a pending message if there is none yet). Then (in both cases) it suspends until next method completion or suspension.

Class SuspendActivity is defined as a subclass of class ConcurrentActivity as it defines launching of sub activities processes.

Instance Variables:

	suspendedSubProcessAssociationList 	<OrderedCollection>
							holds the ordered collection of waiting (suspended) subprocesses.
							Each item of the collection is an association:
								<condition block , semaphore on which the process waits>.
	controlSemaphore		<Semaphore>	is the main control semaphore to synchronize the main activity process
							and the starting/suspension/resumption/completion of subprocesses created.'!


!SuspendActivity methodsFor: 'initialize'!

privateInitialize
	super privateInitialize.
	suspendedSubProcessAssociationList := OrderedCollection new.
	controlSemaphore := Semaphore new! !

!SuspendActivity methodsFor: 'activity setting'!

body
	"The behavior body specifies the main control activity body.
	If there is a suspended message whose resumption condition has become true, resume it.
	Otherwise create a new subprocess for next pending message.
	Then current main activity process gets suspended (until next method completion or suspension)."

	| association |
	[true] whileTrue:
		["First look for (the first) suspended subprocess whose resumption condition has become true."
		association := self lookForFirstResumeableSubProcess.
		association isNil
			ifFalse:
				["If found, then resume it."
				self resumeSuspendedSubProcessAssociation: association]
			ifTrue:
				["Otherwise, accept next message (by starting a new subprocess)."
				self acceptNextMessage].
		"In both cases, wait for resumption (after next method completion or suspension)."
		controlSemaphore wait]! !

!SuspendActivity methodsFor: 'suspension management'!

addSuspendedSubProcessOnCondition: resumptionConditionBlock andSemaphore: aSemaphore
	"Create and add one association for a given subprocess resumption condition and waiting semaphore."

	"association = <key: resumption condition block , value: semaphore on which the process waits>."

	suspendedSubProcessAssociationList addLast:
		(Association new
			key: resumptionConditionBlock
			value: aSemaphore)!

lookForFirstResumeableSubProcess
	"Look for the first suspended subprocess whose resumption condition has become true.
	If one is found, return the corresponding association, otherwise return nil."

	"association = <key: resumption condition block , value: semaphore on which the process waits>."

	^suspendedSubProcessAssociationList isEmpty
		ifTrue:
			[nil]
		ifFalse:
			[suspendedSubProcessAssociationList	
				"Detect if there is a suspended subprocess whose resumption condition block value is true."
				detect: [:association | association key value]
				ifNone: [nil]]!

resumeSuspendedSubProcessAssociation: association
	"Resume a suspended subprocess by removing the association from the list and signaling the semaphore on which it is waiting."

	"association = <key: resumption condition block , value: semaphore on which the process waits>."

	suspendedSubProcessAssociationList remove: association.
	association value signal!

subProcessWaitUntil: resumptionConditionBlock
	"Suspend current method computation on a resumption condition (block)."
	"Equivalent to relinquish: construct in ConcurrentSmalltalk-II."

	| aSemaphore |
	aSemaphore := Semaphore new.
	"Create and add one association for current subprocess suspension to be done."
	self addSuspendedSubProcessOnCondition: resumptionConditionBlock andSemaphore: aSemaphore.
	"First resume the main activity body process."
	controlSemaphore signal.
	"Then, suspend current subprocess onto the semaphore."
	aSemaphore wait

	"Note that when signaled, the suspended subprocess resumes here,
	and immediately returns to resumption of processing the method body which had been suspended."! !

!SuspendActivity methodsFor: 'events'!

kernelEventComplete: aMessage
	"Resume the main control activity process after completing activation of a message."

	super kernelEventComplete: aMessage.
	controlSemaphore signal! !

!SuspendActivity methodsFor: 'process control'!

terminate
	"Terminate all suspended subprocesses at first."

	"association = <key: resumption condition block , value: semaphore on which the process waits>."
	suspendedSubProcessAssociationList do: [:association |
		"For each suspension semaphore, terminate the waiting process."
		association value terminateProcess].
	super terminate! !

SuspendActivity subclass: #ConcurrentSmalltalkActivity
	instanceVariableNames: 'postBlockDictionary '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Suspend'!
ConcurrentSmalltalkActivity comment:
'Activity class ConcurrentSmalltalkActivity provides post actions which can be executed after returning the value, thus freeing the sender from unneeded waiting.
It is defined as a subclass of activity class SuspendActivity.

Instance Variables:

	postBlockDictionary	<Dictionary>	the dictionary containing post actions blocks specified by currently computing sub activities.

As opposed to PoolActivity simple implementation of post actions through a single variable postBlock used as a flag and a container, we are now using a dictionary.
This is because method computation may be suspended before completion and consequently several post actions may be specified before one method completes.
Therefore we must be able to link a block of post actions with the corresponding method computation which specified it. We use the subprocess computing the method as a key. The dictionary stores the following associations:
	key:		a subprocess,
	value:	a post actions block.'!


!ConcurrentSmalltalkActivity methodsFor: 'initialize'!

privateInitialize
	super privateInitialize.
	postBlockDictionary := Dictionary new! !

!ConcurrentSmalltalkActivity methodsFor: 'accessing'!

postBlock: aBlock
	"Add the post actions block into the dictionary at current (sub) process key."

	postBlockDictionary
		at: Processor activeProcess
		put: aBlock! !

!ConcurrentSmalltalkActivity methodsFor: 'events'!

kernelEventComplete: aMessage
	"Check if some post action attached to current method computation has been specified.
	If the case, compute it just after performing the message (before final completion)."

	| postBlock |
	postBlock := postBlockDictionary at: Processor activeProcess ifAbsent: [nil].
	postBlock isNil
		ifFalse:
			[postBlock value.
			postBlockDictionary removeKey: Processor activeProcess].
	super kernelEventComplete: aMessage! !

SuspendActivity subclass: #ConcurrentSmalltalk2Activity
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Suspend'!
ConcurrentSmalltalk2Activity comment:
'Activity class ConcurrentSmalltalk2Activity provides an alternative implementation of post actions.
It is defined as a subclass of activity class SuspendActivity.

In this alternative model, post actions are defined as full methods, prefixed by postOF.
See example ConcurrentSmalltalk2BoundedBuffer.
Note that this model does not allow a plain method and its post method to share temporary variables because they have distinct bodies.'!


!ConcurrentSmalltalk2Activity methodsFor: 'events'!

kernelEventComplete: aMessage
	"Compute the post action method if existing."

	| postSelector |
	postSelector := ('postOF' , aMessage selector) asSymbol.
	(bself respondsTo: postSelector)
		ifTrue: [bself perform: postSelector withArguments: aMessage arguments].
	super kernelEventComplete: aMessage! !

SuspendObject subclass: #ConcurrentSmalltalkObject
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Actalk-Synchro-Suspend'!
ConcurrentSmalltalkObject comment:
'Class ConcurrentSmalltalkObject implements post actions.
It is defined as a subclass of behavior class SuspendObject.'!


!ConcurrentSmalltalkObject methodsFor: 'post actions'!

post: aBlock
	"Specify post actions (specified within a block) which are to be computed
	after completing the method and its return.
	This frees the sender from unneeded waiting, as post actions will take place after return.
	Typical use is for reinitialization."

	self activity postBlock: aBlock! !

!ConcurrentSmalltalkObject methodsFor: 'default classes'!

activityClass
	"ConcurrentSmalltalkObject user construct method
		post:
			calls
	ConcurrentSmalltalkActivity (or ConcurrentSmalltalk2Activity) method
		postBlock:"

	^ConcurrentSmalltalkActivity! !

!ConcurrentSmalltalkObject methodsFor: 'compatibility constraints'!

activityConstraint
	"Enforce constraint defined in method activityClass."

	^Array
		with: ConcurrentSmalltalkActivity
		with: ConcurrentSmalltalk2Activity! !

